/*
Copyright 2024 Manfred Peter

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <QCoreApplication>
#include <QStringList>
#include <QDebug>
#include <QFileInfo>
#include <QCommandLineParser>

#include "htlc.h"

/**
 * Command line parsing is done with CLI11: https://github.com/CLIUtils/CLI11
 */
int main ( int argc, char *argv[] )
{
  QCoreApplication a ( argc, argv );

  QCoreApplication::setApplicationName ( PROGNAME );
  QCoreApplication::setApplicationVersion ( VERSION );

  QCommandLineParser parser;
  QString appDescription = QString::asprintf ( "%s - Wget Link Checker and Corrector\nVersion %s\n(c)%s Manfred Peter\n", PROGNAME, VERSION, YEAR );
  parser.setApplicationDescription ( appDescription );
  parser.addHelpOption();
  parser.addVersionOption();

  // Required
  QCommandLineOption baseDirectoryOption ( QStringList() << "d" << "base_directory",
                                           "(Required) Directory where Wget downloaded the files.",
                                           "dir" );
  parser.addOption ( baseDirectoryOption );

  // Required
  QCommandLineOption serverNamesOption ( QStringList() << "s" << "server_names",
                                         "(Required) Names of servers from which Wget downloaded the files (multi valued).",
                                         "name" );
  parser.addOption ( serverNamesOption );

  // Optional
  QCommandLineOption outputOption ( QStringList() << "o" << "output",
                                    "(Optional) Write output to this file instead of stdout.",
                                    "file" );
  parser.addOption ( outputOption );

  // Optional
  QCommandLineOption missingLinksOption ( QStringList() << "m" << "missing_links",
                                          "(Optional) Write missing links to this file.",
                                          "file" );
  parser.addOption ( missingLinksOption );

  // Optional
  QCommandLineOption charMappingsOption ( QStringList() << "c" << "character_mapping",
                                          "(Optional) Replace characters in links. See manual for syntax (multi valued).",
                                          "string" );
  parser.addOption ( charMappingsOption );

  // Optional
  QCommandLineOption fixLinksOption ( QStringList() << "f" << "fix_links",
                                      "(Optional) If missing link targets were found:\n- Cleanup links\n- Make absolute links relative\n- Backup files\n Only links with the given keyword are fixed.\n(Multi valued)",
                                      "keyword" );
  parser.addOption ( fixLinksOption );

  // Optional
  QCommandLineOption errorsOnlyOption ( QStringList() << "e" << "errors_only",
                                        "(Optional) Report only errors." );
  parser.addOption ( errorsOnlyOption );


  // Process command line options
  parser.process ( a );

  QStringList info;
  QString bar = QString ( "\u2500" ).repeated ( 160 );

  info << parser.applicationDescription() << "";

  QString baseDirectory;
  if ( parser.isSet ( baseDirectoryOption ) )
  {
    baseDirectory = parser.value ( baseDirectoryOption );
    baseDirectory = QFileInfo ( baseDirectory ).absoluteFilePath();
    info << baseDirectoryOption.description() << bar << baseDirectoryOption.names()[1] + " = " + baseDirectory << bar << "\n";
  }
  else
  {
    qCritical() << "Option -d, --base_directory is required";
    exit ( EXIT_FAILURE );
  }

  QStringList downloadServers;
  if ( parser.isSet ( serverNamesOption ) )
  {
    downloadServers = parser.values ( serverNamesOption );
    info << serverNamesOption.description() << bar << serverNamesOption.names()[1] + " = " + downloadServers.join ( "," ) << bar << "\n";
  }
  else
  {
    qCritical() << "Option -s, --server_names is required";
    exit ( EXIT_FAILURE );
  }

  QString outputFile = parser.value ( outputOption );
  info << outputOption.description() << bar << outputOption.names()[1] + " = " + outputFile << bar << "\n";

  QStringList fixLinks = parser.values ( fixLinksOption );
  info << fixLinksOption.description() << bar << fixLinksOption.names()[1] + " = " + fixLinks.join ( "," ) << bar << "\n";

  bool errorsOnly = parser.isSet ( errorsOnlyOption );
  info << errorsOnlyOption.description() << bar << errorsOnlyOption.names()[1] + " = " + QString ( errorsOnly ? "true" : "false" ) << bar << "\n";

  QString missingLinksFile = parser.value ( missingLinksOption );
  info << missingLinksOption.description() << bar << missingLinksOption.names()[1] + " = " + missingLinksFile << bar << "\n";

  QStringList charMappings = parser.values ( charMappingsOption );
  info << charMappingsOption.description() << bar << charMappingsOption.names()[1] + " = " + charMappings.join ( "," ) << bar << "\n";


  // Check and correct links

  Htlc linkChecker ( baseDirectory, downloadServers, outputFile, missingLinksFile, charMappings, fixLinks, errorsOnly, info );

  linkChecker.execute();
}
