/*
Copyright 2024 Manfred Peter

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef HTLC_H
#define HTLC_H

#include <QStringList>
#include <QTextStream>

#define TARGET_DOWNLOADED      "[TARGET DOWNLOADED]       "
#define TARGET_NOT_DOWNLOADED  "[TARGET NOT DOWNLOADED]   "
#define CODEC                  "[CODEC]                   "
#define INTERNAL_ERROR         "[INTERNAL ERROR]          "
#define CHECKING_FILE          "[CHECKING FILE]           "
#define LINK_NEEDS_CORRECTION  "[LINK NEEDS CORRECTION]   "
#define LINK_IGNORED           "[LINK IGNORED]            "
#define PARAMETER_ERROR        "[PARAMETER ERROR]         "
#define FIXING_FILE            "[FIXING FILE]             "
#define SIMILAR_TARGET         "[SIMILAR TARGET]          "

typedef QPair<QString,QString> Mapping;
typedef QList<Mapping> Mappings;

class Htlc
{
  public:
    Htlc ( QString baseDirectory, QStringList downloadServers, QString outputFile,
           QString missingLinksFile, QStringList charMappings,  QStringList fixLinks, bool errorsOnly, QStringList info );
    virtual ~Htlc();
    void execute();

  private:
    QString                m_baseDirectory;            // Where Wget stored the downloaded files. Absolute path without trailing '/'.
    QStringList            m_downloadServers;          // From where Wget downloaded the files
    QString                m_outputFile;               // Output file
    QString                m_missingLinksFile;         // Missings link targets are written to this file
    Mappings               m_charMappings;             // Mapping of sequences of characters
    QStringList            m_fixLinks;                 // List of keywords for which the links shall be converted
    bool                   m_errorsOnly;               // True = Report only errors
    QStringList            m_otherServers;             // Other servers referenced in links
    QStringList            m_fileList;                 // List of files to process
    QMultiHash<QString,QString> m_pathsForFile;        // Maps a filename without path to all paths where a file with this name exists
    int                    m_relativeLinks;            // Number or relative links
    int                    m_numLinksOtherServers;     // Number of absolute links to other servers
    int                    m_numLinksDownloadServers;  // Number of absolute links to downloaded servers
    QRegExp                m_regExpHref;               // Regexp to detect links in HTML files
    QRegExp                m_regExpEntity;             // Regexp to detect entities in filenames
    QTextStream            *m_output;                  // Output stream
    QTextStream            *m_missingLinks;            // Missing links stream
    QHash<QString,QString> m_namedEntities;            // Maps HTML entity names to Unicode characters


    void        getFileList();
    void        checkFile ( QString path );
    /*QStringList*/ void getFiles ( QString path );
    QStringList getDirectories ( QString path );
    QString     createRelativeLink ( QString target, QString source );
    void        checkTargetExists ( QString targetFileName , bool &exists, QString &actualFileName );
    void        getLinkComponents ( QString link, QString &server, QString &path );
    bool        ignoreLink ( QString link );
    QString     cleanupLink ( QString link );
    QString     replaceHTMLEntities ( QString string );
    void        initEntityMap();
    QStringList findFile ( QString fileName );
};

#endif // HTLC_H
