/*
Copyright 2024 Manfred Peter

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "htlc.h"

#include <QDir>
#include <QDebug>
#include <QTextCodec>
#include <QUrl>
//#include <QTextDocumentFragment>

/*
inline QTextStream& qStdout()
{
  static QTextStream stream ( stdout );
  return stream;
}
*/

Htlc::Htlc ( QString baseDirectory, QStringList downloadServers , QString outputFile ,
             QString missingLinksFile, QStringList charMappings, QStringList fixLinks, bool errorsOnly, QStringList info )
{
  m_baseDirectory           = baseDirectory;
  m_downloadServers         = downloadServers;
  m_outputFile              = outputFile;
  m_missingLinksFile        = missingLinksFile;
  m_fixLinks                = fixLinks;
  m_errorsOnly              = errorsOnly;
  m_relativeLinks           = 0;
  m_numLinksOtherServers    = 0;
  m_numLinksDownloadServers = 0;

  if ( outputFile.isEmpty() )
  {
    m_output = new QTextStream ( stdout );
  }
  else
  {
    QFile *file = new QFile ( outputFile );
    file->open ( QIODevice::WriteOnly );
    m_output = new QTextStream ( file );
  }

  for ( QString line : info )
  {
    *m_output << line << "\n";
  }

  if ( missingLinksFile.isEmpty() )
  {
    m_missingLinks = nullptr;
  }
  else
  {
    QFile *file = new QFile ( missingLinksFile );
    file->open ( QIODevice::WriteOnly );
    m_missingLinks = new QTextStream ( file );
  }

  for ( QString charMapping : charMappings )
  {
    int pos = charMapping.indexOf ( "/" );

    if ( pos >= 0 )
    {
      Mapping mapping ( charMapping.left ( pos ), charMapping.mid ( pos + 1 ) );
      m_charMappings.append ( mapping );
    }
    else
    {
      *m_output << PARAMETER_ERROR << "Invalid character mapping (option -c): " << charMapping << "\n";
    }
  }

  m_regExpHref = QRegExp ( "(href|src)[ \t\n\r]*=[ \t\n\r]*(\"([^\"]+)\"|'([^']+)')", Qt::CaseInsensitive );

  m_regExpEntity = QRegExp ( "&(#([0-9]+)|#[xX]([a-fA-F0-9]+)|([a-zA-Z]+));", Qt::CaseSensitive );

  initEntityMap();

  getFileList();
}

Htlc::~Htlc()
{
  m_output->flush();
  delete m_output;

  if ( m_missingLinks )
  {
    m_missingLinks->flush();
    delete m_missingLinks;
  }
}

/**
 * @brief WgetLinkChecker::getDirectories returns all directories in directoryPath
 * @param path
 * @return
 */
QStringList Htlc::getDirectories ( QString directoryPath )
{
  QDir dir ( directoryPath );

  QFileInfoList infoList = dir.entryInfoList ( QDir::AllDirs | QDir::NoDotAndDotDot );

  QStringList dirList;

  foreach ( QFileInfo fileInfo, infoList )
  {
    dirList.append ( fileInfo.filePath() );
  }

  return dirList;
}

/**
 * @brief WgetLinkChecker::getFiles returns all files with names ending with "htm" or "html" in directoryPath
 * @param directoryPath
 * @return List of files
 */
/*QStringList*/ void Htlc::getFiles ( QString directoryPath )
{
  QDir dir ( directoryPath );

//  QStringList nameFilters;
//  // The filter is case insensitive!!
//  nameFilters.append ( "*htm" );
//  nameFilters.append ( "*html" );

//  QFileInfoList infoList = dir.entryInfoList ( nameFilters, QDir::Files );
  QFileInfoList infoList = dir.entryInfoList ( QDir::Files );

  QStringList fileList;

  foreach ( QFileInfo fileInfo, infoList )
  {
    QString filename = fileInfo.fileName();

    if ( filename.endsWith ( "htm", Qt::CaseInsensitive ) || filename.endsWith ( "html", Qt::CaseInsensitive ) )
    {
      m_fileList.append ( fileInfo.filePath() );
    }
    else
    {
      m_pathsForFile.insert ( filename, fileInfo.filePath() );
    }
  }

//  return fileList;
}

/**
 * @brief WgetLinkChecker::getFileList gets the list of files with names ending with "htm" or "html" in the base directory and below
 */
void Htlc::getFileList()
{
  QStringList dirList;

  dirList.append ( m_baseDirectory );

  while ( !dirList.isEmpty() )
  {
    QString dir = dirList.takeLast();
    getFiles ( dir );
 //   m_fileList.append ( getFiles ( dir ) );
    dirList.append ( getDirectories ( dir ) );
  }
}


QString Htlc::cleanupLink ( QString link )
{
  // Remove percent encoding
  QByteArray bytes = link.toUtf8();
  QString cleanLink = QUrl::fromPercentEncoding ( bytes );

  cleanLink = replaceHTMLEntities ( cleanLink );

  // Apply character mapping

  for ( Mapping charMapping : m_charMappings )
  {
    cleanLink = cleanLink.replace ( charMapping.first, charMapping.second );
  }

  return cleanLink;
}

/**
 * @brief WgetLinkChecker::checkTargetExists checks whether a file with the name targetFileName exists in the file system. If not it applies some
 * heuristics to find the file. The name under which the file is found is returned in actualFileName.
 * @param targetFileName Name of the file as it appears in the link
 * @param exists True if target file exists
 * @param actualFileName Actual name of the file in the file system if it exists
 */
void Htlc::checkTargetExists ( QString targetFileName, bool &exists, QString &actualFileName )
{
  exists = QFile::exists ( targetFileName ) ;

  if ( exists )
  {
    actualFileName = targetFileName;
  }
  else
  {
    // Apply heuristics

    actualFileName = cleanupLink ( targetFileName );

    exists = QFile::exists ( actualFileName ) ;

    if ( !exists )
    {
      actualFileName = "";
    }
  }
}

void Htlc::getLinkComponents ( QString link, QString &server, QString &path )
{
  if ( link.startsWith ( "http://" ) || link.startsWith ( "https://" ) )  // Absolute link
  {
    // Absolute link
    int pos = link.indexOf ( "//" );    // Must be >= 0
    QString mainPart = link.mid ( pos + 2 );
    pos = mainPart.indexOf ( "/" );
    if ( pos >= 0 )
    {
      server = mainPart.left ( pos );
      path = mainPart.mid ( pos + 1 );
    }
    else
    {
      server = mainPart;
      path = "";
    }
  }
  else  // Relative link
  {
    server = "";
    path = link;
  }
}

bool Htlc::ignoreLink ( QString link )
{
  if ( link.startsWith ( "mailto:" ) || link.startsWith ( "javascript:" ) || link.startsWith ( "file:" ) )
  {
    return true;
  }
  else
  {
    return false;
  }

}


QStringList Htlc::findFile ( QString fileName )
{
  int pos = fileName.lastIndexOf ( "/" );

  QString name = fileName.mid ( pos + 1 );

  return m_pathsForFile.values ( name );
}

void Htlc::checkFile ( QString fileName )
{
  if ( !m_errorsOnly )
  {
    QStringList info;
    QString bar = QString ( "\u2500" ).repeated ( 160 );
    *m_output << bar << "\n";
    *m_output << CHECKING_FILE << fileName << '\n';
  }

  // Consistency check

  if ( !fileName.startsWith ( m_baseDirectory ) )
  {
    *m_output << INTERNAL_ERROR << "File not in the base directory\n";
    exit ( -1 );
  }

  // Read file

  QFile file ( fileName );

  if ( !file.open ( QIODevice::ReadOnly | QIODevice::Text ) )
  {
    *m_output << INTERNAL_ERROR << "Cannot open file " << fileName << "\n";
    exit ( -1 );
  }

  QByteArray bytes = file.readAll();

  file.close();

  QTextCodec *codec = QTextCodec::codecForHtml ( bytes );

  QString text = codec->toUnicode ( bytes );

  if ( !m_errorsOnly )
  {
    *m_output << CODEC << codec->name() << '\n';
  }

  // Loop at all links in the file

  QString filePath = QFileInfo ( fileName ).absolutePath();

  int pos = 0;
  bool linkChanged = false;
  QString newText;

  while ( ( m_regExpHref.indexIn ( text, pos ) ) != -1 )
  {
    // Get link
    // Capture group 3 or 4 matches

    QString link = m_regExpHref.cap ( 3 );
    int posLink = m_regExpHref.pos ( 3 );

    if ( link.isEmpty() )
    {
      link = m_regExpHref.cap ( 4 );
      posLink = m_regExpHref.pos ( 4 );
    }

    // Store text between links

    newText += text.mid ( pos, posLink - pos );

    // Move pointer to position after link

    pos = posLink + link.length();

    // Check whether link is excluded from the check

    if ( ignoreLink ( link ) )
    {
      if ( !m_errorsOnly )
      {
        *m_output << LINK_IGNORED << link << '\n';
      }
      newText += link;
      continue;
    }

    // Check whether link is relative or absolute
    // Check whether target was downloaded

    QString server;
    QString path;
    getLinkComponents ( link, server, path );

    if ( server.isEmpty() ) // Relative link
    {
      bool exists;
      QString actualFileName;
      QString targetFilename = filePath + "/" + path;
      checkTargetExists ( targetFilename, exists, actualFileName );
      if ( exists )
      {
        if ( actualFileName != targetFilename )
        {
          *m_output << LINK_NEEDS_CORRECTION << link;

          // Create relative link
          QString relativeLink = createRelativeLink ( actualFileName, fileName );

          *m_output << " --> " << relativeLink << '\n';

          if ( m_fixLinks.contains ( QString ( LINK_NEEDS_CORRECTION ).trimmed() ) )
          {
            newText += relativeLink;
            linkChanged = true;
          }
          else
          {
            newText += link;
          }
        }
        else
        {
          if ( !m_errorsOnly )
          {
            *m_output << TARGET_DOWNLOADED << link << '\n';
          }
          newText += link;
        }
      }
      else
      {
        // Check whether a file with the same name exists in another directory
        QStringList hits = findFile ( path );

        if ( !hits.empty() )
        {
          for ( QString hit : hits )
          {
            QString relativeLink = createRelativeLink ( hit, fileName );
            *m_output << SIMILAR_TARGET << link << " --> " << relativeLink << '\n';
          }

          if ( m_fixLinks.contains ( QString ( SIMILAR_TARGET ).trimmed() ) )
          {
            QString relativeLink = createRelativeLink ( hits[0], fileName );
            newText += relativeLink;
            linkChanged = true;
          }
          else
          {
            *m_output << TARGET_NOT_DOWNLOADED << link << '\n';
            newText += link;
          }
        }
        else
        {
          *m_output << TARGET_NOT_DOWNLOADED << link << '\n';
          newText += link;
        }
      }

      m_relativeLinks++;
    }
    else  // Absolute link
    {
      if ( m_downloadServers.contains ( server ) )  // server is in the list of download servers
      {
        m_numLinksDownloadServers++;

        // Check if target was downloaded

        bool exists;
        QString actualFileName;
        QString targetFileName = m_baseDirectory + "/" + server + "/" + path;
        checkTargetExists ( targetFileName, exists, actualFileName );

        if ( exists )
        {
          // Create relative link

          QString relativeLink = createRelativeLink ( actualFileName, fileName );

          *m_output << LINK_NEEDS_CORRECTION << link << " --> " << relativeLink << '\n';

          if ( m_fixLinks.contains ( QString ( LINK_NEEDS_CORRECTION ).trimmed() ) )
          {
            newText += relativeLink;
            linkChanged = true;
          }
          else
          {
            newText += link;
          }
        }
        else
        {
          // Check whether a file with the same name exists in another directory
          QStringList hits = findFile ( path );

          if ( !hits.isEmpty() )
          {
            for ( QString hit : hits )
            {
              QString relativeLink = createRelativeLink ( hit, fileName );
              *m_output << SIMILAR_TARGET << link << " --> " << relativeLink << '\n';
            }

            if ( m_fixLinks.contains ( QString ( SIMILAR_TARGET ).trimmed() ) )
            {
              QString relativeLink = createRelativeLink ( hits[0], fileName );
              newText += relativeLink;
              linkChanged = true;
            }
            else
            {
              newText += link;
            }
          }
          else
          {
            *m_output << TARGET_NOT_DOWNLOADED << link << '\n';
            newText += link;
          }

          if ( m_missingLinks )
          {
            *m_missingLinks << cleanupLink ( link ) << "\n";
          }
        }
      }
      else  // server is not in the list of download servers: Do not check or change link
      {
        m_numLinksOtherServers++;
        m_otherServers.append ( server );

        newText += link;
      }
    }

  }

  // Append remaining text without links

  newText += text.mid ( pos );


  if ( linkChanged )
  {
    // Find a new filename that is not yet used

    //QFile::remove ( fileName + ".bak" );

    QString backupName = fileName + ".bak";
    if ( QFile::exists ( backupName ) )
    {
      int counter = 1;
      while ( QFile::exists ( backupName + "." + QString::number ( counter ) ) ) { counter++; }
      backupName += "." + QString::number ( counter );
    }

    bool success = QFile::rename ( fileName, backupName );

    if ( success )
    {
      QFile newFile ( fileName );

      if ( !newFile.open ( QIODevice::WriteOnly | QIODevice::Text ) )
      {
        *m_output << INTERNAL_ERROR << "Cannot write to file " << fileName << "\n";
        exit ( -1 );
      }

      bytes = codec->fromUnicode ( newText );
      newFile.write ( bytes );
      newFile.close();
      *m_output << FIXING_FILE << fileName << '\n';
    }
    else
    {
      *m_output << INTERNAL_ERROR << "Cannot create backup of file " << fileName << "\n";
      exit ( -1 );
    }
  }
}

void Htlc::execute()
{
  foreach ( QString file, m_fileList )
  {
    checkFile ( file );
  }

  if ( !m_errorsOnly )
  {
    m_otherServers.sort();
    m_otherServers.removeDuplicates();

    *m_output << '\n';
    *m_output << "===========================================================================================\n";
    *m_output << "===========================================================================================\n";
    *m_output << "===== Other Hosts =========================================================================\n";
    *m_output << "===========================================================================================\n";
    *m_output << "===========================================================================================\n";
    *m_output << '\n';

    foreach ( QString host, m_otherServers )
    {
      *m_output << host << '\n';
    }

    *m_output << '\n';
    *m_output << "===========================================================================================\n";
    *m_output << "===========================================================================================\n";
    *m_output << "===== Statistics ==========================================================================\n";
    *m_output << "===========================================================================================\n";
    *m_output << "===========================================================================================\n";
    *m_output << '\n';

    *m_output << "Number of HTML files: " << m_fileList.size() << '\n';
    *m_output << "Relative Links: " << m_relativeLinks << '\n';
    *m_output << "Absolute Links included: " << m_numLinksDownloadServers << '\n';
    *m_output << "Absolute Links excluded: " << m_numLinksOtherServers << '\n';
  }
}

QString Htlc::createRelativeLink ( QString target, QString source )
{
  QString relativeTarget = target;
  QString relativeSource = source;

  bool commonDir = true;
  int index = 0;

  while ( ( index = relativeSource.indexOf ( '/' ) ) >= 0 )
  {
    if ( commonDir && relativeTarget.startsWith ( relativeSource.left ( index ) + "/" ) )
    {
      relativeTarget = relativeTarget.mid ( index + 1 );
    }
    else
    {
      commonDir = false;
      relativeTarget = "../" + relativeTarget;
    }

    relativeSource = relativeSource.mid ( index + 1 );
  }

  return relativeTarget;

}

// We use our own method because QTextDocumentFragment::fromHtml ( string ).toPlainText() makes other changes
// to the text besides replacing HTML entities. It also replaces consecutive whitespace by one single space.
QString Htlc::replaceHTMLEntities ( QString string )
{
  QString result;
  int pos = 0;

  while ( m_regExpEntity.indexIn ( string, pos ) != -1 )
  {
    int posEntity = m_regExpEntity.pos ( 0 );

    // Add text between entities
    result += string.mid ( pos, posEntity - pos );

    QString decimal = m_regExpEntity.cap ( 2 ) ;
    QString hex = m_regExpEntity.cap ( 3 ) ;
    QString name = m_regExpEntity.cap ( 4 ) ;

    QChar c;
    bool ok = false;

    if ( !decimal.isEmpty() )
    {
      c = QChar ( decimal.toInt ( &ok, 10 ) );
    }
    else if ( !hex.isEmpty() )
    {
      c = QChar ( hex.toInt ( &ok, 16 ) );
    }
    else // In this case name cannot be empty
    {
      if ( m_namedEntities.contains ( name ) )
      {
        c = QChar ( m_namedEntities.value ( name ).toInt ( &ok, 16 ) );
      }
      else
      {
        ok = false;
      }

    }

    if ( ok )
    {
      result += c;
    }
    else
    {
      result += m_regExpEntity.cap ( 0 );
    }

    pos = posEntity + m_regExpEntity.matchedLength();

  }

  // Add text after the last entity
  result += string.mid ( pos );

  return result;
}



// This mapping was created from the JSON file entities.json at
//   https://html.spec.whatwg.org/entities.json
// with the command
//   sed -nE -f script.sed entities.json > mapping.txt
// where script.sed contains the line
//  s/[^&]*&([a-zA-Z]+);.*\\u([0-9A-F]+)".*/  m_namedEntities.insert ( "\1", "\2" );/p
void Htlc::initEntityMap()
{
  m_namedEntities.insert ( "AElig", "00C6" );
  m_namedEntities.insert ( "AMP", "0026" );
  m_namedEntities.insert ( "Aacute", "00C1" );
  m_namedEntities.insert ( "Abreve", "0102" );
  m_namedEntities.insert ( "Acirc", "00C2" );
  m_namedEntities.insert ( "Acy", "0410" );
  m_namedEntities.insert ( "Afr", "DD04" );
  m_namedEntities.insert ( "Agrave", "00C0" );
  m_namedEntities.insert ( "Alpha", "0391" );
  m_namedEntities.insert ( "Amacr", "0100" );
  m_namedEntities.insert ( "And", "2A53" );
  m_namedEntities.insert ( "Aogon", "0104" );
  m_namedEntities.insert ( "Aopf", "DD38" );
  m_namedEntities.insert ( "ApplyFunction", "2061" );
  m_namedEntities.insert ( "Aring", "00C5" );
  m_namedEntities.insert ( "Ascr", "DC9C" );
  m_namedEntities.insert ( "Assign", "2254" );
  m_namedEntities.insert ( "Atilde", "00C3" );
  m_namedEntities.insert ( "Auml", "00C4" );
  m_namedEntities.insert ( "Backslash", "2216" );
  m_namedEntities.insert ( "Barv", "2AE7" );
  m_namedEntities.insert ( "Barwed", "2306" );
  m_namedEntities.insert ( "Bcy", "0411" );
  m_namedEntities.insert ( "Because", "2235" );
  m_namedEntities.insert ( "Bernoullis", "212C" );
  m_namedEntities.insert ( "Beta", "0392" );
  m_namedEntities.insert ( "Bfr", "DD05" );
  m_namedEntities.insert ( "Bopf", "DD39" );
  m_namedEntities.insert ( "Breve", "02D8" );
  m_namedEntities.insert ( "Bscr", "212C" );
  m_namedEntities.insert ( "Bumpeq", "224E" );
  m_namedEntities.insert ( "CHcy", "0427" );
  m_namedEntities.insert ( "COPY", "00A9" );
  m_namedEntities.insert ( "Cacute", "0106" );
  m_namedEntities.insert ( "Cap", "22D2" );
  m_namedEntities.insert ( "CapitalDifferentialD", "2145" );
  m_namedEntities.insert ( "Cayleys", "212D" );
  m_namedEntities.insert ( "Ccaron", "010C" );
  m_namedEntities.insert ( "Ccedil", "00C7" );
  m_namedEntities.insert ( "Ccirc", "0108" );
  m_namedEntities.insert ( "Cconint", "2230" );
  m_namedEntities.insert ( "Cdot", "010A" );
  m_namedEntities.insert ( "Cedilla", "00B8" );
  m_namedEntities.insert ( "CenterDot", "00B7" );
  m_namedEntities.insert ( "Cfr", "212D" );
  m_namedEntities.insert ( "Chi", "03A7" );
  m_namedEntities.insert ( "CircleDot", "2299" );
  m_namedEntities.insert ( "CircleMinus", "2296" );
  m_namedEntities.insert ( "CirclePlus", "2295" );
  m_namedEntities.insert ( "CircleTimes", "2297" );
  m_namedEntities.insert ( "ClockwiseContourIntegral", "2232" );
  m_namedEntities.insert ( "CloseCurlyDoubleQuote", "201D" );
  m_namedEntities.insert ( "CloseCurlyQuote", "2019" );
  m_namedEntities.insert ( "Colon", "2237" );
  m_namedEntities.insert ( "Colone", "2A74" );
  m_namedEntities.insert ( "Congruent", "2261" );
  m_namedEntities.insert ( "Conint", "222F" );
  m_namedEntities.insert ( "ContourIntegral", "222E" );
  m_namedEntities.insert ( "Copf", "2102" );
  m_namedEntities.insert ( "Coproduct", "2210" );
  m_namedEntities.insert ( "CounterClockwiseContourIntegral", "2233" );
  m_namedEntities.insert ( "Cross", "2A2F" );
  m_namedEntities.insert ( "Cscr", "DC9E" );
  m_namedEntities.insert ( "Cup", "22D3" );
  m_namedEntities.insert ( "CupCap", "224D" );
  m_namedEntities.insert ( "DD", "2145" );
  m_namedEntities.insert ( "DDotrahd", "2911" );
  m_namedEntities.insert ( "DJcy", "0402" );
  m_namedEntities.insert ( "DScy", "0405" );
  m_namedEntities.insert ( "DZcy", "040F" );
  m_namedEntities.insert ( "Dagger", "2021" );
  m_namedEntities.insert ( "Darr", "21A1" );
  m_namedEntities.insert ( "Dashv", "2AE4" );
  m_namedEntities.insert ( "Dcaron", "010E" );
  m_namedEntities.insert ( "Dcy", "0414" );
  m_namedEntities.insert ( "Del", "2207" );
  m_namedEntities.insert ( "Delta", "0394" );
  m_namedEntities.insert ( "Dfr", "DD07" );
  m_namedEntities.insert ( "DiacriticalAcute", "00B4" );
  m_namedEntities.insert ( "DiacriticalDot", "02D9" );
  m_namedEntities.insert ( "DiacriticalDoubleAcute", "02DD" );
  m_namedEntities.insert ( "DiacriticalGrave", "0060" );
  m_namedEntities.insert ( "DiacriticalTilde", "02DC" );
  m_namedEntities.insert ( "Diamond", "22C4" );
  m_namedEntities.insert ( "DifferentialD", "2146" );
  m_namedEntities.insert ( "Dopf", "DD3B" );
  m_namedEntities.insert ( "Dot", "00A8" );
  m_namedEntities.insert ( "DotDot", "20DC" );
  m_namedEntities.insert ( "DotEqual", "2250" );
  m_namedEntities.insert ( "DoubleContourIntegral", "222F" );
  m_namedEntities.insert ( "DoubleDot", "00A8" );
  m_namedEntities.insert ( "DoubleDownArrow", "21D3" );
  m_namedEntities.insert ( "DoubleLeftArrow", "21D0" );
  m_namedEntities.insert ( "DoubleLeftRightArrow", "21D4" );
  m_namedEntities.insert ( "DoubleLeftTee", "2AE4" );
  m_namedEntities.insert ( "DoubleLongLeftArrow", "27F8" );
  m_namedEntities.insert ( "DoubleLongLeftRightArrow", "27FA" );
  m_namedEntities.insert ( "DoubleLongRightArrow", "27F9" );
  m_namedEntities.insert ( "DoubleRightArrow", "21D2" );
  m_namedEntities.insert ( "DoubleRightTee", "22A8" );
  m_namedEntities.insert ( "DoubleUpArrow", "21D1" );
  m_namedEntities.insert ( "DoubleUpDownArrow", "21D5" );
  m_namedEntities.insert ( "DoubleVerticalBar", "2225" );
  m_namedEntities.insert ( "DownArrow", "2193" );
  m_namedEntities.insert ( "DownArrowBar", "2913" );
  m_namedEntities.insert ( "DownArrowUpArrow", "21F5" );
  m_namedEntities.insert ( "DownBreve", "0311" );
  m_namedEntities.insert ( "DownLeftRightVector", "2950" );
  m_namedEntities.insert ( "DownLeftTeeVector", "295E" );
  m_namedEntities.insert ( "DownLeftVector", "21BD" );
  m_namedEntities.insert ( "DownLeftVectorBar", "2956" );
  m_namedEntities.insert ( "DownRightTeeVector", "295F" );
  m_namedEntities.insert ( "DownRightVector", "21C1" );
  m_namedEntities.insert ( "DownRightVectorBar", "2957" );
  m_namedEntities.insert ( "DownTee", "22A4" );
  m_namedEntities.insert ( "DownTeeArrow", "21A7" );
  m_namedEntities.insert ( "Downarrow", "21D3" );
  m_namedEntities.insert ( "Dscr", "DC9F" );
  m_namedEntities.insert ( "Dstrok", "0110" );
  m_namedEntities.insert ( "ENG", "014A" );
  m_namedEntities.insert ( "ETH", "00D0" );
  m_namedEntities.insert ( "Eacute", "00C9" );
  m_namedEntities.insert ( "Ecaron", "011A" );
  m_namedEntities.insert ( "Ecirc", "00CA" );
  m_namedEntities.insert ( "Ecy", "042D" );
  m_namedEntities.insert ( "Edot", "0116" );
  m_namedEntities.insert ( "Efr", "DD08" );
  m_namedEntities.insert ( "Egrave", "00C8" );
  m_namedEntities.insert ( "Element", "2208" );
  m_namedEntities.insert ( "Emacr", "0112" );
  m_namedEntities.insert ( "EmptySmallSquare", "25FB" );
  m_namedEntities.insert ( "EmptyVerySmallSquare", "25AB" );
  m_namedEntities.insert ( "Eogon", "0118" );
  m_namedEntities.insert ( "Eopf", "DD3C" );
  m_namedEntities.insert ( "Epsilon", "0395" );
  m_namedEntities.insert ( "Equal", "2A75" );
  m_namedEntities.insert ( "EqualTilde", "2242" );
  m_namedEntities.insert ( "Equilibrium", "21CC" );
  m_namedEntities.insert ( "Escr", "2130" );
  m_namedEntities.insert ( "Esim", "2A73" );
  m_namedEntities.insert ( "Eta", "0397" );
  m_namedEntities.insert ( "Euml", "00CB" );
  m_namedEntities.insert ( "Exists", "2203" );
  m_namedEntities.insert ( "ExponentialE", "2147" );
  m_namedEntities.insert ( "Fcy", "0424" );
  m_namedEntities.insert ( "Ffr", "DD09" );
  m_namedEntities.insert ( "FilledSmallSquare", "25FC" );
  m_namedEntities.insert ( "FilledVerySmallSquare", "25AA" );
  m_namedEntities.insert ( "Fopf", "DD3D" );
  m_namedEntities.insert ( "ForAll", "2200" );
  m_namedEntities.insert ( "Fouriertrf", "2131" );
  m_namedEntities.insert ( "Fscr", "2131" );
  m_namedEntities.insert ( "GJcy", "0403" );
  m_namedEntities.insert ( "GT", "003E" );
  m_namedEntities.insert ( "Gamma", "0393" );
  m_namedEntities.insert ( "Gammad", "03DC" );
  m_namedEntities.insert ( "Gbreve", "011E" );
  m_namedEntities.insert ( "Gcedil", "0122" );
  m_namedEntities.insert ( "Gcirc", "011C" );
  m_namedEntities.insert ( "Gcy", "0413" );
  m_namedEntities.insert ( "Gdot", "0120" );
  m_namedEntities.insert ( "Gfr", "DD0A" );
  m_namedEntities.insert ( "Gg", "22D9" );
  m_namedEntities.insert ( "Gopf", "DD3E" );
  m_namedEntities.insert ( "GreaterEqual", "2265" );
  m_namedEntities.insert ( "GreaterEqualLess", "22DB" );
  m_namedEntities.insert ( "GreaterFullEqual", "2267" );
  m_namedEntities.insert ( "GreaterGreater", "2AA2" );
  m_namedEntities.insert ( "GreaterLess", "2277" );
  m_namedEntities.insert ( "GreaterSlantEqual", "2A7E" );
  m_namedEntities.insert ( "GreaterTilde", "2273" );
  m_namedEntities.insert ( "Gscr", "DCA2" );
  m_namedEntities.insert ( "Gt", "226B" );
  m_namedEntities.insert ( "HARDcy", "042A" );
  m_namedEntities.insert ( "Hacek", "02C7" );
  m_namedEntities.insert ( "Hat", "005E" );
  m_namedEntities.insert ( "Hcirc", "0124" );
  m_namedEntities.insert ( "Hfr", "210C" );
  m_namedEntities.insert ( "HilbertSpace", "210B" );
  m_namedEntities.insert ( "Hopf", "210D" );
  m_namedEntities.insert ( "HorizontalLine", "2500" );
  m_namedEntities.insert ( "Hscr", "210B" );
  m_namedEntities.insert ( "Hstrok", "0126" );
  m_namedEntities.insert ( "HumpDownHump", "224E" );
  m_namedEntities.insert ( "HumpEqual", "224F" );
  m_namedEntities.insert ( "IEcy", "0415" );
  m_namedEntities.insert ( "IJlig", "0132" );
  m_namedEntities.insert ( "IOcy", "0401" );
  m_namedEntities.insert ( "Iacute", "00CD" );
  m_namedEntities.insert ( "Icirc", "00CE" );
  m_namedEntities.insert ( "Icy", "0418" );
  m_namedEntities.insert ( "Idot", "0130" );
  m_namedEntities.insert ( "Ifr", "2111" );
  m_namedEntities.insert ( "Igrave", "00CC" );
  m_namedEntities.insert ( "Im", "2111" );
  m_namedEntities.insert ( "Imacr", "012A" );
  m_namedEntities.insert ( "ImaginaryI", "2148" );
  m_namedEntities.insert ( "Implies", "21D2" );
  m_namedEntities.insert ( "Int", "222C" );
  m_namedEntities.insert ( "Integral", "222B" );
  m_namedEntities.insert ( "Intersection", "22C2" );
  m_namedEntities.insert ( "InvisibleComma", "2063" );
  m_namedEntities.insert ( "InvisibleTimes", "2062" );
  m_namedEntities.insert ( "Iogon", "012E" );
  m_namedEntities.insert ( "Iopf", "DD40" );
  m_namedEntities.insert ( "Iota", "0399" );
  m_namedEntities.insert ( "Iscr", "2110" );
  m_namedEntities.insert ( "Itilde", "0128" );
  m_namedEntities.insert ( "Iukcy", "0406" );
  m_namedEntities.insert ( "Iuml", "00CF" );
  m_namedEntities.insert ( "Jcirc", "0134" );
  m_namedEntities.insert ( "Jcy", "0419" );
  m_namedEntities.insert ( "Jfr", "DD0D" );
  m_namedEntities.insert ( "Jopf", "DD41" );
  m_namedEntities.insert ( "Jscr", "DCA5" );
  m_namedEntities.insert ( "Jsercy", "0408" );
  m_namedEntities.insert ( "Jukcy", "0404" );
  m_namedEntities.insert ( "KHcy", "0425" );
  m_namedEntities.insert ( "KJcy", "040C" );
  m_namedEntities.insert ( "Kappa", "039A" );
  m_namedEntities.insert ( "Kcedil", "0136" );
  m_namedEntities.insert ( "Kcy", "041A" );
  m_namedEntities.insert ( "Kfr", "DD0E" );
  m_namedEntities.insert ( "Kopf", "DD42" );
  m_namedEntities.insert ( "Kscr", "DCA6" );
  m_namedEntities.insert ( "LJcy", "0409" );
  m_namedEntities.insert ( "LT", "003C" );
  m_namedEntities.insert ( "Lacute", "0139" );
  m_namedEntities.insert ( "Lambda", "039B" );
  m_namedEntities.insert ( "Lang", "27EA" );
  m_namedEntities.insert ( "Laplacetrf", "2112" );
  m_namedEntities.insert ( "Larr", "219E" );
  m_namedEntities.insert ( "Lcaron", "013D" );
  m_namedEntities.insert ( "Lcedil", "013B" );
  m_namedEntities.insert ( "Lcy", "041B" );
  m_namedEntities.insert ( "LeftAngleBracket", "27E8" );
  m_namedEntities.insert ( "LeftArrow", "2190" );
  m_namedEntities.insert ( "LeftArrowBar", "21E4" );
  m_namedEntities.insert ( "LeftArrowRightArrow", "21C6" );
  m_namedEntities.insert ( "LeftCeiling", "2308" );
  m_namedEntities.insert ( "LeftDoubleBracket", "27E6" );
  m_namedEntities.insert ( "LeftDownTeeVector", "2961" );
  m_namedEntities.insert ( "LeftDownVector", "21C3" );
  m_namedEntities.insert ( "LeftDownVectorBar", "2959" );
  m_namedEntities.insert ( "LeftFloor", "230A" );
  m_namedEntities.insert ( "LeftRightArrow", "2194" );
  m_namedEntities.insert ( "LeftRightVector", "294E" );
  m_namedEntities.insert ( "LeftTee", "22A3" );
  m_namedEntities.insert ( "LeftTeeArrow", "21A4" );
  m_namedEntities.insert ( "LeftTeeVector", "295A" );
  m_namedEntities.insert ( "LeftTriangle", "22B2" );
  m_namedEntities.insert ( "LeftTriangleBar", "29CF" );
  m_namedEntities.insert ( "LeftTriangleEqual", "22B4" );
  m_namedEntities.insert ( "LeftUpDownVector", "2951" );
  m_namedEntities.insert ( "LeftUpTeeVector", "2960" );
  m_namedEntities.insert ( "LeftUpVector", "21BF" );
  m_namedEntities.insert ( "LeftUpVectorBar", "2958" );
  m_namedEntities.insert ( "LeftVector", "21BC" );
  m_namedEntities.insert ( "LeftVectorBar", "2952" );
  m_namedEntities.insert ( "Leftarrow", "21D0" );
  m_namedEntities.insert ( "Leftrightarrow", "21D4" );
  m_namedEntities.insert ( "LessEqualGreater", "22DA" );
  m_namedEntities.insert ( "LessFullEqual", "2266" );
  m_namedEntities.insert ( "LessGreater", "2276" );
  m_namedEntities.insert ( "LessLess", "2AA1" );
  m_namedEntities.insert ( "LessSlantEqual", "2A7D" );
  m_namedEntities.insert ( "LessTilde", "2272" );
  m_namedEntities.insert ( "Lfr", "DD0F" );
  m_namedEntities.insert ( "Ll", "22D8" );
  m_namedEntities.insert ( "Lleftarrow", "21DA" );
  m_namedEntities.insert ( "Lmidot", "013F" );
  m_namedEntities.insert ( "LongLeftArrow", "27F5" );
  m_namedEntities.insert ( "LongLeftRightArrow", "27F7" );
  m_namedEntities.insert ( "LongRightArrow", "27F6" );
  m_namedEntities.insert ( "Longleftarrow", "27F8" );
  m_namedEntities.insert ( "Longleftrightarrow", "27FA" );
  m_namedEntities.insert ( "Longrightarrow", "27F9" );
  m_namedEntities.insert ( "Lopf", "DD43" );
  m_namedEntities.insert ( "LowerLeftArrow", "2199" );
  m_namedEntities.insert ( "LowerRightArrow", "2198" );
  m_namedEntities.insert ( "Lscr", "2112" );
  m_namedEntities.insert ( "Lsh", "21B0" );
  m_namedEntities.insert ( "Lstrok", "0141" );
  m_namedEntities.insert ( "Lt", "226A" );
  m_namedEntities.insert ( "Map", "2905" );
  m_namedEntities.insert ( "Mcy", "041C" );
  m_namedEntities.insert ( "MediumSpace", "205F" );
  m_namedEntities.insert ( "Mellintrf", "2133" );
  m_namedEntities.insert ( "Mfr", "DD10" );
  m_namedEntities.insert ( "MinusPlus", "2213" );
  m_namedEntities.insert ( "Mopf", "DD44" );
  m_namedEntities.insert ( "Mscr", "2133" );
  m_namedEntities.insert ( "Mu", "039C" );
  m_namedEntities.insert ( "NJcy", "040A" );
  m_namedEntities.insert ( "Nacute", "0143" );
  m_namedEntities.insert ( "Ncaron", "0147" );
  m_namedEntities.insert ( "Ncedil", "0145" );
  m_namedEntities.insert ( "Ncy", "041D" );
  m_namedEntities.insert ( "NegativeMediumSpace", "200B" );
  m_namedEntities.insert ( "NegativeThickSpace", "200B" );
  m_namedEntities.insert ( "NegativeThinSpace", "200B" );
  m_namedEntities.insert ( "NegativeVeryThinSpace", "200B" );
  m_namedEntities.insert ( "NestedGreaterGreater", "226B" );
  m_namedEntities.insert ( "NestedLessLess", "226A" );
  m_namedEntities.insert ( "NewLine", "000A" );
  m_namedEntities.insert ( "Nfr", "DD11" );
  m_namedEntities.insert ( "NoBreak", "2060" );
  m_namedEntities.insert ( "NonBreakingSpace", "00A0" );
  m_namedEntities.insert ( "Nopf", "2115" );
  m_namedEntities.insert ( "Not", "2AEC" );
  m_namedEntities.insert ( "NotCongruent", "2262" );
  m_namedEntities.insert ( "NotCupCap", "226D" );
  m_namedEntities.insert ( "NotDoubleVerticalBar", "2226" );
  m_namedEntities.insert ( "NotElement", "2209" );
  m_namedEntities.insert ( "NotEqual", "2260" );
  m_namedEntities.insert ( "NotEqualTilde", "0338" );
  m_namedEntities.insert ( "NotExists", "2204" );
  m_namedEntities.insert ( "NotGreater", "226F" );
  m_namedEntities.insert ( "NotGreaterEqual", "2271" );
  m_namedEntities.insert ( "NotGreaterFullEqual", "0338" );
  m_namedEntities.insert ( "NotGreaterGreater", "0338" );
  m_namedEntities.insert ( "NotGreaterLess", "2279" );
  m_namedEntities.insert ( "NotGreaterSlantEqual", "0338" );
  m_namedEntities.insert ( "NotGreaterTilde", "2275" );
  m_namedEntities.insert ( "NotHumpDownHump", "0338" );
  m_namedEntities.insert ( "NotHumpEqual", "0338" );
  m_namedEntities.insert ( "NotLeftTriangle", "22EA" );
  m_namedEntities.insert ( "NotLeftTriangleBar", "0338" );
  m_namedEntities.insert ( "NotLeftTriangleEqual", "22EC" );
  m_namedEntities.insert ( "NotLess", "226E" );
  m_namedEntities.insert ( "NotLessEqual", "2270" );
  m_namedEntities.insert ( "NotLessGreater", "2278" );
  m_namedEntities.insert ( "NotLessLess", "0338" );
  m_namedEntities.insert ( "NotLessSlantEqual", "0338" );
  m_namedEntities.insert ( "NotLessTilde", "2274" );
  m_namedEntities.insert ( "NotNestedGreaterGreater", "0338" );
  m_namedEntities.insert ( "NotNestedLessLess", "0338" );
  m_namedEntities.insert ( "NotPrecedes", "2280" );
  m_namedEntities.insert ( "NotPrecedesEqual", "0338" );
  m_namedEntities.insert ( "NotPrecedesSlantEqual", "22E0" );
  m_namedEntities.insert ( "NotReverseElement", "220C" );
  m_namedEntities.insert ( "NotRightTriangle", "22EB" );
  m_namedEntities.insert ( "NotRightTriangleBar", "0338" );
  m_namedEntities.insert ( "NotRightTriangleEqual", "22ED" );
  m_namedEntities.insert ( "NotSquareSubset", "0338" );
  m_namedEntities.insert ( "NotSquareSubsetEqual", "22E2" );
  m_namedEntities.insert ( "NotSquareSuperset", "0338" );
  m_namedEntities.insert ( "NotSquareSupersetEqual", "22E3" );
  m_namedEntities.insert ( "NotSubset", "20D2" );
  m_namedEntities.insert ( "NotSubsetEqual", "2288" );
  m_namedEntities.insert ( "NotSucceeds", "2281" );
  m_namedEntities.insert ( "NotSucceedsEqual", "0338" );
  m_namedEntities.insert ( "NotSucceedsSlantEqual", "22E1" );
  m_namedEntities.insert ( "NotSucceedsTilde", "0338" );
  m_namedEntities.insert ( "NotSuperset", "20D2" );
  m_namedEntities.insert ( "NotSupersetEqual", "2289" );
  m_namedEntities.insert ( "NotTilde", "2241" );
  m_namedEntities.insert ( "NotTildeEqual", "2244" );
  m_namedEntities.insert ( "NotTildeFullEqual", "2247" );
  m_namedEntities.insert ( "NotTildeTilde", "2249" );
  m_namedEntities.insert ( "NotVerticalBar", "2224" );
  m_namedEntities.insert ( "Nscr", "DCA9" );
  m_namedEntities.insert ( "Ntilde", "00D1" );
  m_namedEntities.insert ( "Nu", "039D" );
  m_namedEntities.insert ( "OElig", "0152" );
  m_namedEntities.insert ( "Oacute", "00D3" );
  m_namedEntities.insert ( "Ocirc", "00D4" );
  m_namedEntities.insert ( "Ocy", "041E" );
  m_namedEntities.insert ( "Odblac", "0150" );
  m_namedEntities.insert ( "Ofr", "DD12" );
  m_namedEntities.insert ( "Ograve", "00D2" );
  m_namedEntities.insert ( "Omacr", "014C" );
  m_namedEntities.insert ( "Omega", "03A9" );
  m_namedEntities.insert ( "Omicron", "039F" );
  m_namedEntities.insert ( "Oopf", "DD46" );
  m_namedEntities.insert ( "OpenCurlyDoubleQuote", "201C" );
  m_namedEntities.insert ( "OpenCurlyQuote", "2018" );
  m_namedEntities.insert ( "Or", "2A54" );
  m_namedEntities.insert ( "Oscr", "DCAA" );
  m_namedEntities.insert ( "Oslash", "00D8" );
  m_namedEntities.insert ( "Otilde", "00D5" );
  m_namedEntities.insert ( "Otimes", "2A37" );
  m_namedEntities.insert ( "Ouml", "00D6" );
  m_namedEntities.insert ( "OverBar", "203E" );
  m_namedEntities.insert ( "OverBrace", "23DE" );
  m_namedEntities.insert ( "OverBracket", "23B4" );
  m_namedEntities.insert ( "OverParenthesis", "23DC" );
  m_namedEntities.insert ( "PartialD", "2202" );
  m_namedEntities.insert ( "Pcy", "041F" );
  m_namedEntities.insert ( "Pfr", "DD13" );
  m_namedEntities.insert ( "Phi", "03A6" );
  m_namedEntities.insert ( "Pi", "03A0" );
  m_namedEntities.insert ( "PlusMinus", "00B1" );
  m_namedEntities.insert ( "Poincareplane", "210C" );
  m_namedEntities.insert ( "Popf", "2119" );
  m_namedEntities.insert ( "Pr", "2ABB" );
  m_namedEntities.insert ( "Precedes", "227A" );
  m_namedEntities.insert ( "PrecedesEqual", "2AAF" );
  m_namedEntities.insert ( "PrecedesSlantEqual", "227C" );
  m_namedEntities.insert ( "PrecedesTilde", "227E" );
  m_namedEntities.insert ( "Prime", "2033" );
  m_namedEntities.insert ( "Product", "220F" );
  m_namedEntities.insert ( "Proportion", "2237" );
  m_namedEntities.insert ( "Proportional", "221D" );
  m_namedEntities.insert ( "Pscr", "DCAB" );
  m_namedEntities.insert ( "Psi", "03A8" );
  m_namedEntities.insert ( "QUOT", "0022" );
  m_namedEntities.insert ( "Qfr", "DD14" );
  m_namedEntities.insert ( "Qopf", "211A" );
  m_namedEntities.insert ( "Qscr", "DCAC" );
  m_namedEntities.insert ( "RBarr", "2910" );
  m_namedEntities.insert ( "REG", "00AE" );
  m_namedEntities.insert ( "Racute", "0154" );
  m_namedEntities.insert ( "Rang", "27EB" );
  m_namedEntities.insert ( "Rarr", "21A0" );
  m_namedEntities.insert ( "Rarrtl", "2916" );
  m_namedEntities.insert ( "Rcaron", "0158" );
  m_namedEntities.insert ( "Rcedil", "0156" );
  m_namedEntities.insert ( "Rcy", "0420" );
  m_namedEntities.insert ( "Re", "211C" );
  m_namedEntities.insert ( "ReverseElement", "220B" );
  m_namedEntities.insert ( "ReverseEquilibrium", "21CB" );
  m_namedEntities.insert ( "ReverseUpEquilibrium", "296F" );
  m_namedEntities.insert ( "Rfr", "211C" );
  m_namedEntities.insert ( "Rho", "03A1" );
  m_namedEntities.insert ( "RightAngleBracket", "27E9" );
  m_namedEntities.insert ( "RightArrow", "2192" );
  m_namedEntities.insert ( "RightArrowBar", "21E5" );
  m_namedEntities.insert ( "RightArrowLeftArrow", "21C4" );
  m_namedEntities.insert ( "RightCeiling", "2309" );
  m_namedEntities.insert ( "RightDoubleBracket", "27E7" );
  m_namedEntities.insert ( "RightDownTeeVector", "295D" );
  m_namedEntities.insert ( "RightDownVector", "21C2" );
  m_namedEntities.insert ( "RightDownVectorBar", "2955" );
  m_namedEntities.insert ( "RightFloor", "230B" );
  m_namedEntities.insert ( "RightTee", "22A2" );
  m_namedEntities.insert ( "RightTeeArrow", "21A6" );
  m_namedEntities.insert ( "RightTeeVector", "295B" );
  m_namedEntities.insert ( "RightTriangle", "22B3" );
  m_namedEntities.insert ( "RightTriangleBar", "29D0" );
  m_namedEntities.insert ( "RightTriangleEqual", "22B5" );
  m_namedEntities.insert ( "RightUpDownVector", "294F" );
  m_namedEntities.insert ( "RightUpTeeVector", "295C" );
  m_namedEntities.insert ( "RightUpVector", "21BE" );
  m_namedEntities.insert ( "RightUpVectorBar", "2954" );
  m_namedEntities.insert ( "RightVector", "21C0" );
  m_namedEntities.insert ( "RightVectorBar", "2953" );
  m_namedEntities.insert ( "Rightarrow", "21D2" );
  m_namedEntities.insert ( "Ropf", "211D" );
  m_namedEntities.insert ( "RoundImplies", "2970" );
  m_namedEntities.insert ( "Rrightarrow", "21DB" );
  m_namedEntities.insert ( "Rscr", "211B" );
  m_namedEntities.insert ( "Rsh", "21B1" );
  m_namedEntities.insert ( "RuleDelayed", "29F4" );
  m_namedEntities.insert ( "SHCHcy", "0429" );
  m_namedEntities.insert ( "SHcy", "0428" );
  m_namedEntities.insert ( "SOFTcy", "042C" );
  m_namedEntities.insert ( "Sacute", "015A" );
  m_namedEntities.insert ( "Sc", "2ABC" );
  m_namedEntities.insert ( "Scaron", "0160" );
  m_namedEntities.insert ( "Scedil", "015E" );
  m_namedEntities.insert ( "Scirc", "015C" );
  m_namedEntities.insert ( "Scy", "0421" );
  m_namedEntities.insert ( "Sfr", "DD16" );
  m_namedEntities.insert ( "ShortDownArrow", "2193" );
  m_namedEntities.insert ( "ShortLeftArrow", "2190" );
  m_namedEntities.insert ( "ShortRightArrow", "2192" );
  m_namedEntities.insert ( "ShortUpArrow", "2191" );
  m_namedEntities.insert ( "Sigma", "03A3" );
  m_namedEntities.insert ( "SmallCircle", "2218" );
  m_namedEntities.insert ( "Sopf", "DD4A" );
  m_namedEntities.insert ( "Sqrt", "221A" );
  m_namedEntities.insert ( "Square", "25A1" );
  m_namedEntities.insert ( "SquareIntersection", "2293" );
  m_namedEntities.insert ( "SquareSubset", "228F" );
  m_namedEntities.insert ( "SquareSubsetEqual", "2291" );
  m_namedEntities.insert ( "SquareSuperset", "2290" );
  m_namedEntities.insert ( "SquareSupersetEqual", "2292" );
  m_namedEntities.insert ( "SquareUnion", "2294" );
  m_namedEntities.insert ( "Sscr", "DCAE" );
  m_namedEntities.insert ( "Star", "22C6" );
  m_namedEntities.insert ( "Sub", "22D0" );
  m_namedEntities.insert ( "Subset", "22D0" );
  m_namedEntities.insert ( "SubsetEqual", "2286" );
  m_namedEntities.insert ( "Succeeds", "227B" );
  m_namedEntities.insert ( "SucceedsEqual", "2AB0" );
  m_namedEntities.insert ( "SucceedsSlantEqual", "227D" );
  m_namedEntities.insert ( "SucceedsTilde", "227F" );
  m_namedEntities.insert ( "SuchThat", "220B" );
  m_namedEntities.insert ( "Sum", "2211" );
  m_namedEntities.insert ( "Sup", "22D1" );
  m_namedEntities.insert ( "Superset", "2283" );
  m_namedEntities.insert ( "SupersetEqual", "2287" );
  m_namedEntities.insert ( "Supset", "22D1" );
  m_namedEntities.insert ( "THORN", "00DE" );
  m_namedEntities.insert ( "TRADE", "2122" );
  m_namedEntities.insert ( "TSHcy", "040B" );
  m_namedEntities.insert ( "TScy", "0426" );
  m_namedEntities.insert ( "Tab", "0009" );
  m_namedEntities.insert ( "Tau", "03A4" );
  m_namedEntities.insert ( "Tcaron", "0164" );
  m_namedEntities.insert ( "Tcedil", "0162" );
  m_namedEntities.insert ( "Tcy", "0422" );
  m_namedEntities.insert ( "Tfr", "DD17" );
  m_namedEntities.insert ( "Therefore", "2234" );
  m_namedEntities.insert ( "Theta", "0398" );
  m_namedEntities.insert ( "ThickSpace", "200A" );
  m_namedEntities.insert ( "ThinSpace", "2009" );
  m_namedEntities.insert ( "Tilde", "223C" );
  m_namedEntities.insert ( "TildeEqual", "2243" );
  m_namedEntities.insert ( "TildeFullEqual", "2245" );
  m_namedEntities.insert ( "TildeTilde", "2248" );
  m_namedEntities.insert ( "Topf", "DD4B" );
  m_namedEntities.insert ( "TripleDot", "20DB" );
  m_namedEntities.insert ( "Tscr", "DCAF" );
  m_namedEntities.insert ( "Tstrok", "0166" );
  m_namedEntities.insert ( "Uacute", "00DA" );
  m_namedEntities.insert ( "Uarr", "219F" );
  m_namedEntities.insert ( "Uarrocir", "2949" );
  m_namedEntities.insert ( "Ubrcy", "040E" );
  m_namedEntities.insert ( "Ubreve", "016C" );
  m_namedEntities.insert ( "Ucirc", "00DB" );
  m_namedEntities.insert ( "Ucy", "0423" );
  m_namedEntities.insert ( "Udblac", "0170" );
  m_namedEntities.insert ( "Ufr", "DD18" );
  m_namedEntities.insert ( "Ugrave", "00D9" );
  m_namedEntities.insert ( "Umacr", "016A" );
  m_namedEntities.insert ( "UnderBar", "005F" );
  m_namedEntities.insert ( "UnderBrace", "23DF" );
  m_namedEntities.insert ( "UnderBracket", "23B5" );
  m_namedEntities.insert ( "UnderParenthesis", "23DD" );
  m_namedEntities.insert ( "Union", "22C3" );
  m_namedEntities.insert ( "UnionPlus", "228E" );
  m_namedEntities.insert ( "Uogon", "0172" );
  m_namedEntities.insert ( "Uopf", "DD4C" );
  m_namedEntities.insert ( "UpArrow", "2191" );
  m_namedEntities.insert ( "UpArrowBar", "2912" );
  m_namedEntities.insert ( "UpArrowDownArrow", "21C5" );
  m_namedEntities.insert ( "UpDownArrow", "2195" );
  m_namedEntities.insert ( "UpEquilibrium", "296E" );
  m_namedEntities.insert ( "UpTee", "22A5" );
  m_namedEntities.insert ( "UpTeeArrow", "21A5" );
  m_namedEntities.insert ( "Uparrow", "21D1" );
  m_namedEntities.insert ( "Updownarrow", "21D5" );
  m_namedEntities.insert ( "UpperLeftArrow", "2196" );
  m_namedEntities.insert ( "UpperRightArrow", "2197" );
  m_namedEntities.insert ( "Upsi", "03D2" );
  m_namedEntities.insert ( "Upsilon", "03A5" );
  m_namedEntities.insert ( "Uring", "016E" );
  m_namedEntities.insert ( "Uscr", "DCB0" );
  m_namedEntities.insert ( "Utilde", "0168" );
  m_namedEntities.insert ( "Uuml", "00DC" );
  m_namedEntities.insert ( "VDash", "22AB" );
  m_namedEntities.insert ( "Vbar", "2AEB" );
  m_namedEntities.insert ( "Vcy", "0412" );
  m_namedEntities.insert ( "Vdash", "22A9" );
  m_namedEntities.insert ( "Vdashl", "2AE6" );
  m_namedEntities.insert ( "Vee", "22C1" );
  m_namedEntities.insert ( "Verbar", "2016" );
  m_namedEntities.insert ( "Vert", "2016" );
  m_namedEntities.insert ( "VerticalBar", "2223" );
  m_namedEntities.insert ( "VerticalLine", "007C" );
  m_namedEntities.insert ( "VerticalSeparator", "2758" );
  m_namedEntities.insert ( "VerticalTilde", "2240" );
  m_namedEntities.insert ( "VeryThinSpace", "200A" );
  m_namedEntities.insert ( "Vfr", "DD19" );
  m_namedEntities.insert ( "Vopf", "DD4D" );
  m_namedEntities.insert ( "Vscr", "DCB1" );
  m_namedEntities.insert ( "Vvdash", "22AA" );
  m_namedEntities.insert ( "Wcirc", "0174" );
  m_namedEntities.insert ( "Wedge", "22C0" );
  m_namedEntities.insert ( "Wfr", "DD1A" );
  m_namedEntities.insert ( "Wopf", "DD4E" );
  m_namedEntities.insert ( "Wscr", "DCB2" );
  m_namedEntities.insert ( "Xfr", "DD1B" );
  m_namedEntities.insert ( "Xi", "039E" );
  m_namedEntities.insert ( "Xopf", "DD4F" );
  m_namedEntities.insert ( "Xscr", "DCB3" );
  m_namedEntities.insert ( "YAcy", "042F" );
  m_namedEntities.insert ( "YIcy", "0407" );
  m_namedEntities.insert ( "YUcy", "042E" );
  m_namedEntities.insert ( "Yacute", "00DD" );
  m_namedEntities.insert ( "Ycirc", "0176" );
  m_namedEntities.insert ( "Ycy", "042B" );
  m_namedEntities.insert ( "Yfr", "DD1C" );
  m_namedEntities.insert ( "Yopf", "DD50" );
  m_namedEntities.insert ( "Yscr", "DCB4" );
  m_namedEntities.insert ( "Yuml", "0178" );
  m_namedEntities.insert ( "ZHcy", "0416" );
  m_namedEntities.insert ( "Zacute", "0179" );
  m_namedEntities.insert ( "Zcaron", "017D" );
  m_namedEntities.insert ( "Zcy", "0417" );
  m_namedEntities.insert ( "Zdot", "017B" );
  m_namedEntities.insert ( "ZeroWidthSpace", "200B" );
  m_namedEntities.insert ( "Zeta", "0396" );
  m_namedEntities.insert ( "Zfr", "2128" );
  m_namedEntities.insert ( "Zopf", "2124" );
  m_namedEntities.insert ( "Zscr", "DCB5" );
  m_namedEntities.insert ( "aacute", "00E1" );
  m_namedEntities.insert ( "abreve", "0103" );
  m_namedEntities.insert ( "ac", "223E" );
  m_namedEntities.insert ( "acE", "0333" );
  m_namedEntities.insert ( "acd", "223F" );
  m_namedEntities.insert ( "acirc", "00E2" );
  m_namedEntities.insert ( "acute", "00B4" );
  m_namedEntities.insert ( "acy", "0430" );
  m_namedEntities.insert ( "aelig", "00E6" );
  m_namedEntities.insert ( "af", "2061" );
  m_namedEntities.insert ( "afr", "DD1E" );
  m_namedEntities.insert ( "agrave", "00E0" );
  m_namedEntities.insert ( "alefsym", "2135" );
  m_namedEntities.insert ( "aleph", "2135" );
  m_namedEntities.insert ( "alpha", "03B1" );
  m_namedEntities.insert ( "amacr", "0101" );
  m_namedEntities.insert ( "amalg", "2A3F" );
  m_namedEntities.insert ( "amp", "0026" );
  m_namedEntities.insert ( "and", "2227" );
  m_namedEntities.insert ( "andand", "2A55" );
  m_namedEntities.insert ( "andd", "2A5C" );
  m_namedEntities.insert ( "andslope", "2A58" );
  m_namedEntities.insert ( "andv", "2A5A" );
  m_namedEntities.insert ( "ang", "2220" );
  m_namedEntities.insert ( "ange", "29A4" );
  m_namedEntities.insert ( "angle", "2220" );
  m_namedEntities.insert ( "angmsd", "2221" );
  m_namedEntities.insert ( "angmsdaa", "29A8" );
  m_namedEntities.insert ( "angmsdab", "29A9" );
  m_namedEntities.insert ( "angmsdac", "29AA" );
  m_namedEntities.insert ( "angmsdad", "29AB" );
  m_namedEntities.insert ( "angmsdae", "29AC" );
  m_namedEntities.insert ( "angmsdaf", "29AD" );
  m_namedEntities.insert ( "angmsdag", "29AE" );
  m_namedEntities.insert ( "angmsdah", "29AF" );
  m_namedEntities.insert ( "angrt", "221F" );
  m_namedEntities.insert ( "angrtvb", "22BE" );
  m_namedEntities.insert ( "angrtvbd", "299D" );
  m_namedEntities.insert ( "angsph", "2222" );
  m_namedEntities.insert ( "angst", "00C5" );
  m_namedEntities.insert ( "angzarr", "237C" );
  m_namedEntities.insert ( "aogon", "0105" );
  m_namedEntities.insert ( "aopf", "DD52" );
  m_namedEntities.insert ( "ap", "2248" );
  m_namedEntities.insert ( "apE", "2A70" );
  m_namedEntities.insert ( "apacir", "2A6F" );
  m_namedEntities.insert ( "ape", "224A" );
  m_namedEntities.insert ( "apid", "224B" );
  m_namedEntities.insert ( "apos", "0027" );
  m_namedEntities.insert ( "approx", "2248" );
  m_namedEntities.insert ( "approxeq", "224A" );
  m_namedEntities.insert ( "aring", "00E5" );
  m_namedEntities.insert ( "ascr", "DCB6" );
  m_namedEntities.insert ( "ast", "002A" );
  m_namedEntities.insert ( "asymp", "2248" );
  m_namedEntities.insert ( "asympeq", "224D" );
  m_namedEntities.insert ( "atilde", "00E3" );
  m_namedEntities.insert ( "auml", "00E4" );
  m_namedEntities.insert ( "awconint", "2233" );
  m_namedEntities.insert ( "awint", "2A11" );
  m_namedEntities.insert ( "bNot", "2AED" );
  m_namedEntities.insert ( "backcong", "224C" );
  m_namedEntities.insert ( "backepsilon", "03F6" );
  m_namedEntities.insert ( "backprime", "2035" );
  m_namedEntities.insert ( "backsim", "223D" );
  m_namedEntities.insert ( "backsimeq", "22CD" );
  m_namedEntities.insert ( "barvee", "22BD" );
  m_namedEntities.insert ( "barwed", "2305" );
  m_namedEntities.insert ( "barwedge", "2305" );
  m_namedEntities.insert ( "bbrk", "23B5" );
  m_namedEntities.insert ( "bbrktbrk", "23B6" );
  m_namedEntities.insert ( "bcong", "224C" );
  m_namedEntities.insert ( "bcy", "0431" );
  m_namedEntities.insert ( "bdquo", "201E" );
  m_namedEntities.insert ( "becaus", "2235" );
  m_namedEntities.insert ( "because", "2235" );
  m_namedEntities.insert ( "bemptyv", "29B0" );
  m_namedEntities.insert ( "bepsi", "03F6" );
  m_namedEntities.insert ( "bernou", "212C" );
  m_namedEntities.insert ( "beta", "03B2" );
  m_namedEntities.insert ( "beth", "2136" );
  m_namedEntities.insert ( "between", "226C" );
  m_namedEntities.insert ( "bfr", "DD1F" );
  m_namedEntities.insert ( "bigcap", "22C2" );
  m_namedEntities.insert ( "bigcirc", "25EF" );
  m_namedEntities.insert ( "bigcup", "22C3" );
  m_namedEntities.insert ( "bigodot", "2A00" );
  m_namedEntities.insert ( "bigoplus", "2A01" );
  m_namedEntities.insert ( "bigotimes", "2A02" );
  m_namedEntities.insert ( "bigsqcup", "2A06" );
  m_namedEntities.insert ( "bigstar", "2605" );
  m_namedEntities.insert ( "bigtriangledown", "25BD" );
  m_namedEntities.insert ( "bigtriangleup", "25B3" );
  m_namedEntities.insert ( "biguplus", "2A04" );
  m_namedEntities.insert ( "bigvee", "22C1" );
  m_namedEntities.insert ( "bigwedge", "22C0" );
  m_namedEntities.insert ( "bkarow", "290D" );
  m_namedEntities.insert ( "blacklozenge", "29EB" );
  m_namedEntities.insert ( "blacksquare", "25AA" );
  m_namedEntities.insert ( "blacktriangle", "25B4" );
  m_namedEntities.insert ( "blacktriangledown", "25BE" );
  m_namedEntities.insert ( "blacktriangleleft", "25C2" );
  m_namedEntities.insert ( "blacktriangleright", "25B8" );
  m_namedEntities.insert ( "blank", "2423" );
  m_namedEntities.insert ( "block", "2588" );
  m_namedEntities.insert ( "bne", "20E5" );
  m_namedEntities.insert ( "bnequiv", "20E5" );
  m_namedEntities.insert ( "bnot", "2310" );
  m_namedEntities.insert ( "bopf", "DD53" );
  m_namedEntities.insert ( "bot", "22A5" );
  m_namedEntities.insert ( "bottom", "22A5" );
  m_namedEntities.insert ( "bowtie", "22C8" );
  m_namedEntities.insert ( "boxDL", "2557" );
  m_namedEntities.insert ( "boxDR", "2554" );
  m_namedEntities.insert ( "boxDl", "2556" );
  m_namedEntities.insert ( "boxDr", "2553" );
  m_namedEntities.insert ( "boxH", "2550" );
  m_namedEntities.insert ( "boxHD", "2566" );
  m_namedEntities.insert ( "boxHU", "2569" );
  m_namedEntities.insert ( "boxHd", "2564" );
  m_namedEntities.insert ( "boxHu", "2567" );
  m_namedEntities.insert ( "boxUL", "255D" );
  m_namedEntities.insert ( "boxUR", "255A" );
  m_namedEntities.insert ( "boxUl", "255C" );
  m_namedEntities.insert ( "boxUr", "2559" );
  m_namedEntities.insert ( "boxV", "2551" );
  m_namedEntities.insert ( "boxVH", "256C" );
  m_namedEntities.insert ( "boxVL", "2563" );
  m_namedEntities.insert ( "boxVR", "2560" );
  m_namedEntities.insert ( "boxVh", "256B" );
  m_namedEntities.insert ( "boxVl", "2562" );
  m_namedEntities.insert ( "boxVr", "255F" );
  m_namedEntities.insert ( "boxbox", "29C9" );
  m_namedEntities.insert ( "boxdL", "2555" );
  m_namedEntities.insert ( "boxdR", "2552" );
  m_namedEntities.insert ( "boxdl", "2510" );
  m_namedEntities.insert ( "boxdr", "250C" );
  m_namedEntities.insert ( "boxh", "2500" );
  m_namedEntities.insert ( "boxhD", "2565" );
  m_namedEntities.insert ( "boxhU", "2568" );
  m_namedEntities.insert ( "boxhd", "252C" );
  m_namedEntities.insert ( "boxhu", "2534" );
  m_namedEntities.insert ( "boxminus", "229F" );
  m_namedEntities.insert ( "boxplus", "229E" );
  m_namedEntities.insert ( "boxtimes", "22A0" );
  m_namedEntities.insert ( "boxuL", "255B" );
  m_namedEntities.insert ( "boxuR", "2558" );
  m_namedEntities.insert ( "boxul", "2518" );
  m_namedEntities.insert ( "boxur", "2514" );
  m_namedEntities.insert ( "boxv", "2502" );
  m_namedEntities.insert ( "boxvH", "256A" );
  m_namedEntities.insert ( "boxvL", "2561" );
  m_namedEntities.insert ( "boxvR", "255E" );
  m_namedEntities.insert ( "boxvh", "253C" );
  m_namedEntities.insert ( "boxvl", "2524" );
  m_namedEntities.insert ( "boxvr", "251C" );
  m_namedEntities.insert ( "bprime", "2035" );
  m_namedEntities.insert ( "breve", "02D8" );
  m_namedEntities.insert ( "brvbar", "00A6" );
  m_namedEntities.insert ( "bscr", "DCB7" );
  m_namedEntities.insert ( "bsemi", "204F" );
  m_namedEntities.insert ( "bsim", "223D" );
  m_namedEntities.insert ( "bsime", "22CD" );
  m_namedEntities.insert ( "bsol", "005C" );
  m_namedEntities.insert ( "bsolb", "29C5" );
  m_namedEntities.insert ( "bsolhsub", "27C8" );
  m_namedEntities.insert ( "bull", "2022" );
  m_namedEntities.insert ( "bullet", "2022" );
  m_namedEntities.insert ( "bump", "224E" );
  m_namedEntities.insert ( "bumpE", "2AAE" );
  m_namedEntities.insert ( "bumpe", "224F" );
  m_namedEntities.insert ( "bumpeq", "224F" );
  m_namedEntities.insert ( "cacute", "0107" );
  m_namedEntities.insert ( "cap", "2229" );
  m_namedEntities.insert ( "capand", "2A44" );
  m_namedEntities.insert ( "capbrcup", "2A49" );
  m_namedEntities.insert ( "capcap", "2A4B" );
  m_namedEntities.insert ( "capcup", "2A47" );
  m_namedEntities.insert ( "capdot", "2A40" );
  m_namedEntities.insert ( "caps", "FE00" );
  m_namedEntities.insert ( "caret", "2041" );
  m_namedEntities.insert ( "caron", "02C7" );
  m_namedEntities.insert ( "ccaps", "2A4D" );
  m_namedEntities.insert ( "ccaron", "010D" );
  m_namedEntities.insert ( "ccedil", "00E7" );
  m_namedEntities.insert ( "ccirc", "0109" );
  m_namedEntities.insert ( "ccups", "2A4C" );
  m_namedEntities.insert ( "ccupssm", "2A50" );
  m_namedEntities.insert ( "cdot", "010B" );
  m_namedEntities.insert ( "cedil", "00B8" );
  m_namedEntities.insert ( "cemptyv", "29B2" );
  m_namedEntities.insert ( "cent", "00A2" );
  m_namedEntities.insert ( "centerdot", "00B7" );
  m_namedEntities.insert ( "cfr", "DD20" );
  m_namedEntities.insert ( "chcy", "0447" );
  m_namedEntities.insert ( "check", "2713" );
  m_namedEntities.insert ( "checkmark", "2713" );
  m_namedEntities.insert ( "chi", "03C7" );
  m_namedEntities.insert ( "cir", "25CB" );
  m_namedEntities.insert ( "cirE", "29C3" );
  m_namedEntities.insert ( "circ", "02C6" );
  m_namedEntities.insert ( "circeq", "2257" );
  m_namedEntities.insert ( "circlearrowleft", "21BA" );
  m_namedEntities.insert ( "circlearrowright", "21BB" );
  m_namedEntities.insert ( "circledR", "00AE" );
  m_namedEntities.insert ( "circledS", "24C8" );
  m_namedEntities.insert ( "circledast", "229B" );
  m_namedEntities.insert ( "circledcirc", "229A" );
  m_namedEntities.insert ( "circleddash", "229D" );
  m_namedEntities.insert ( "cire", "2257" );
  m_namedEntities.insert ( "cirfnint", "2A10" );
  m_namedEntities.insert ( "cirmid", "2AEF" );
  m_namedEntities.insert ( "cirscir", "29C2" );
  m_namedEntities.insert ( "clubs", "2663" );
  m_namedEntities.insert ( "clubsuit", "2663" );
  m_namedEntities.insert ( "colon", "003A" );
  m_namedEntities.insert ( "colone", "2254" );
  m_namedEntities.insert ( "coloneq", "2254" );
  m_namedEntities.insert ( "comma", "002C" );
  m_namedEntities.insert ( "commat", "0040" );
  m_namedEntities.insert ( "comp", "2201" );
  m_namedEntities.insert ( "compfn", "2218" );
  m_namedEntities.insert ( "complement", "2201" );
  m_namedEntities.insert ( "complexes", "2102" );
  m_namedEntities.insert ( "cong", "2245" );
  m_namedEntities.insert ( "congdot", "2A6D" );
  m_namedEntities.insert ( "conint", "222E" );
  m_namedEntities.insert ( "copf", "DD54" );
  m_namedEntities.insert ( "coprod", "2210" );
  m_namedEntities.insert ( "copy", "00A9" );
  m_namedEntities.insert ( "copysr", "2117" );
  m_namedEntities.insert ( "crarr", "21B5" );
  m_namedEntities.insert ( "cross", "2717" );
  m_namedEntities.insert ( "cscr", "DCB8" );
  m_namedEntities.insert ( "csub", "2ACF" );
  m_namedEntities.insert ( "csube", "2AD1" );
  m_namedEntities.insert ( "csup", "2AD0" );
  m_namedEntities.insert ( "csupe", "2AD2" );
  m_namedEntities.insert ( "ctdot", "22EF" );
  m_namedEntities.insert ( "cudarrl", "2938" );
  m_namedEntities.insert ( "cudarrr", "2935" );
  m_namedEntities.insert ( "cuepr", "22DE" );
  m_namedEntities.insert ( "cuesc", "22DF" );
  m_namedEntities.insert ( "cularr", "21B6" );
  m_namedEntities.insert ( "cularrp", "293D" );
  m_namedEntities.insert ( "cup", "222A" );
  m_namedEntities.insert ( "cupbrcap", "2A48" );
  m_namedEntities.insert ( "cupcap", "2A46" );
  m_namedEntities.insert ( "cupcup", "2A4A" );
  m_namedEntities.insert ( "cupdot", "228D" );
  m_namedEntities.insert ( "cupor", "2A45" );
  m_namedEntities.insert ( "cups", "FE00" );
  m_namedEntities.insert ( "curarr", "21B7" );
  m_namedEntities.insert ( "curarrm", "293C" );
  m_namedEntities.insert ( "curlyeqprec", "22DE" );
  m_namedEntities.insert ( "curlyeqsucc", "22DF" );
  m_namedEntities.insert ( "curlyvee", "22CE" );
  m_namedEntities.insert ( "curlywedge", "22CF" );
  m_namedEntities.insert ( "curren", "00A4" );
  m_namedEntities.insert ( "curvearrowleft", "21B6" );
  m_namedEntities.insert ( "curvearrowright", "21B7" );
  m_namedEntities.insert ( "cuvee", "22CE" );
  m_namedEntities.insert ( "cuwed", "22CF" );
  m_namedEntities.insert ( "cwconint", "2232" );
  m_namedEntities.insert ( "cwint", "2231" );
  m_namedEntities.insert ( "cylcty", "232D" );
  m_namedEntities.insert ( "dArr", "21D3" );
  m_namedEntities.insert ( "dHar", "2965" );
  m_namedEntities.insert ( "dagger", "2020" );
  m_namedEntities.insert ( "daleth", "2138" );
  m_namedEntities.insert ( "darr", "2193" );
  m_namedEntities.insert ( "dash", "2010" );
  m_namedEntities.insert ( "dashv", "22A3" );
  m_namedEntities.insert ( "dbkarow", "290F" );
  m_namedEntities.insert ( "dblac", "02DD" );
  m_namedEntities.insert ( "dcaron", "010F" );
  m_namedEntities.insert ( "dcy", "0434" );
  m_namedEntities.insert ( "dd", "2146" );
  m_namedEntities.insert ( "ddagger", "2021" );
  m_namedEntities.insert ( "ddarr", "21CA" );
  m_namedEntities.insert ( "ddotseq", "2A77" );
  m_namedEntities.insert ( "deg", "00B0" );
  m_namedEntities.insert ( "delta", "03B4" );
  m_namedEntities.insert ( "demptyv", "29B1" );
  m_namedEntities.insert ( "dfisht", "297F" );
  m_namedEntities.insert ( "dfr", "DD21" );
  m_namedEntities.insert ( "dharl", "21C3" );
  m_namedEntities.insert ( "dharr", "21C2" );
  m_namedEntities.insert ( "diam", "22C4" );
  m_namedEntities.insert ( "diamond", "22C4" );
  m_namedEntities.insert ( "diamondsuit", "2666" );
  m_namedEntities.insert ( "diams", "2666" );
  m_namedEntities.insert ( "die", "00A8" );
  m_namedEntities.insert ( "digamma", "03DD" );
  m_namedEntities.insert ( "disin", "22F2" );
  m_namedEntities.insert ( "div", "00F7" );
  m_namedEntities.insert ( "divide", "00F7" );
  m_namedEntities.insert ( "divideontimes", "22C7" );
  m_namedEntities.insert ( "divonx", "22C7" );
  m_namedEntities.insert ( "djcy", "0452" );
  m_namedEntities.insert ( "dlcorn", "231E" );
  m_namedEntities.insert ( "dlcrop", "230D" );
  m_namedEntities.insert ( "dollar", "0024" );
  m_namedEntities.insert ( "dopf", "DD55" );
  m_namedEntities.insert ( "dot", "02D9" );
  m_namedEntities.insert ( "doteq", "2250" );
  m_namedEntities.insert ( "doteqdot", "2251" );
  m_namedEntities.insert ( "dotminus", "2238" );
  m_namedEntities.insert ( "dotplus", "2214" );
  m_namedEntities.insert ( "dotsquare", "22A1" );
  m_namedEntities.insert ( "doublebarwedge", "2306" );
  m_namedEntities.insert ( "downarrow", "2193" );
  m_namedEntities.insert ( "downdownarrows", "21CA" );
  m_namedEntities.insert ( "downharpoonleft", "21C3" );
  m_namedEntities.insert ( "downharpoonright", "21C2" );
  m_namedEntities.insert ( "drbkarow", "2910" );
  m_namedEntities.insert ( "drcorn", "231F" );
  m_namedEntities.insert ( "drcrop", "230C" );
  m_namedEntities.insert ( "dscr", "DCB9" );
  m_namedEntities.insert ( "dscy", "0455" );
  m_namedEntities.insert ( "dsol", "29F6" );
  m_namedEntities.insert ( "dstrok", "0111" );
  m_namedEntities.insert ( "dtdot", "22F1" );
  m_namedEntities.insert ( "dtri", "25BF" );
  m_namedEntities.insert ( "dtrif", "25BE" );
  m_namedEntities.insert ( "duarr", "21F5" );
  m_namedEntities.insert ( "duhar", "296F" );
  m_namedEntities.insert ( "dwangle", "29A6" );
  m_namedEntities.insert ( "dzcy", "045F" );
  m_namedEntities.insert ( "dzigrarr", "27FF" );
  m_namedEntities.insert ( "eDDot", "2A77" );
  m_namedEntities.insert ( "eDot", "2251" );
  m_namedEntities.insert ( "eacute", "00E9" );
  m_namedEntities.insert ( "easter", "2A6E" );
  m_namedEntities.insert ( "ecaron", "011B" );
  m_namedEntities.insert ( "ecir", "2256" );
  m_namedEntities.insert ( "ecirc", "00EA" );
  m_namedEntities.insert ( "ecolon", "2255" );
  m_namedEntities.insert ( "ecy", "044D" );
  m_namedEntities.insert ( "edot", "0117" );
  m_namedEntities.insert ( "ee", "2147" );
  m_namedEntities.insert ( "efDot", "2252" );
  m_namedEntities.insert ( "efr", "DD22" );
  m_namedEntities.insert ( "eg", "2A9A" );
  m_namedEntities.insert ( "egrave", "00E8" );
  m_namedEntities.insert ( "egs", "2A96" );
  m_namedEntities.insert ( "egsdot", "2A98" );
  m_namedEntities.insert ( "el", "2A99" );
  m_namedEntities.insert ( "elinters", "23E7" );
  m_namedEntities.insert ( "ell", "2113" );
  m_namedEntities.insert ( "els", "2A95" );
  m_namedEntities.insert ( "elsdot", "2A97" );
  m_namedEntities.insert ( "emacr", "0113" );
  m_namedEntities.insert ( "empty", "2205" );
  m_namedEntities.insert ( "emptyset", "2205" );
  m_namedEntities.insert ( "emptyv", "2205" );
  m_namedEntities.insert ( "emsp", "2003" );
  m_namedEntities.insert ( "eng", "014B" );
  m_namedEntities.insert ( "ensp", "2002" );
  m_namedEntities.insert ( "eogon", "0119" );
  m_namedEntities.insert ( "eopf", "DD56" );
  m_namedEntities.insert ( "epar", "22D5" );
  m_namedEntities.insert ( "eparsl", "29E3" );
  m_namedEntities.insert ( "eplus", "2A71" );
  m_namedEntities.insert ( "epsi", "03B5" );
  m_namedEntities.insert ( "epsilon", "03B5" );
  m_namedEntities.insert ( "epsiv", "03F5" );
  m_namedEntities.insert ( "eqcirc", "2256" );
  m_namedEntities.insert ( "eqcolon", "2255" );
  m_namedEntities.insert ( "eqsim", "2242" );
  m_namedEntities.insert ( "eqslantgtr", "2A96" );
  m_namedEntities.insert ( "eqslantless", "2A95" );
  m_namedEntities.insert ( "equals", "003D" );
  m_namedEntities.insert ( "equest", "225F" );
  m_namedEntities.insert ( "equiv", "2261" );
  m_namedEntities.insert ( "equivDD", "2A78" );
  m_namedEntities.insert ( "eqvparsl", "29E5" );
  m_namedEntities.insert ( "erDot", "2253" );
  m_namedEntities.insert ( "erarr", "2971" );
  m_namedEntities.insert ( "escr", "212F" );
  m_namedEntities.insert ( "esdot", "2250" );
  m_namedEntities.insert ( "esim", "2242" );
  m_namedEntities.insert ( "eta", "03B7" );
  m_namedEntities.insert ( "eth", "00F0" );
  m_namedEntities.insert ( "euml", "00EB" );
  m_namedEntities.insert ( "euro", "20AC" );
  m_namedEntities.insert ( "excl", "0021" );
  m_namedEntities.insert ( "exist", "2203" );
  m_namedEntities.insert ( "expectation", "2130" );
  m_namedEntities.insert ( "exponentiale", "2147" );
  m_namedEntities.insert ( "fallingdotseq", "2252" );
  m_namedEntities.insert ( "fcy", "0444" );
  m_namedEntities.insert ( "female", "2640" );
  m_namedEntities.insert ( "ffilig", "FB03" );
  m_namedEntities.insert ( "fflig", "FB00" );
  m_namedEntities.insert ( "ffllig", "FB04" );
  m_namedEntities.insert ( "ffr", "DD23" );
  m_namedEntities.insert ( "filig", "FB01" );
  m_namedEntities.insert ( "fjlig", "006A" );
  m_namedEntities.insert ( "flat", "266D" );
  m_namedEntities.insert ( "fllig", "FB02" );
  m_namedEntities.insert ( "fltns", "25B1" );
  m_namedEntities.insert ( "fnof", "0192" );
  m_namedEntities.insert ( "fopf", "DD57" );
  m_namedEntities.insert ( "forall", "2200" );
  m_namedEntities.insert ( "fork", "22D4" );
  m_namedEntities.insert ( "forkv", "2AD9" );
  m_namedEntities.insert ( "fpartint", "2A0D" );
  m_namedEntities.insert ( "frasl", "2044" );
  m_namedEntities.insert ( "frown", "2322" );
  m_namedEntities.insert ( "fscr", "DCBB" );
  m_namedEntities.insert ( "gE", "2267" );
  m_namedEntities.insert ( "gEl", "2A8C" );
  m_namedEntities.insert ( "gacute", "01F5" );
  m_namedEntities.insert ( "gamma", "03B3" );
  m_namedEntities.insert ( "gammad", "03DD" );
  m_namedEntities.insert ( "gap", "2A86" );
  m_namedEntities.insert ( "gbreve", "011F" );
  m_namedEntities.insert ( "gcirc", "011D" );
  m_namedEntities.insert ( "gcy", "0433" );
  m_namedEntities.insert ( "gdot", "0121" );
  m_namedEntities.insert ( "ge", "2265" );
  m_namedEntities.insert ( "gel", "22DB" );
  m_namedEntities.insert ( "geq", "2265" );
  m_namedEntities.insert ( "geqq", "2267" );
  m_namedEntities.insert ( "geqslant", "2A7E" );
  m_namedEntities.insert ( "ges", "2A7E" );
  m_namedEntities.insert ( "gescc", "2AA9" );
  m_namedEntities.insert ( "gesdot", "2A80" );
  m_namedEntities.insert ( "gesdoto", "2A82" );
  m_namedEntities.insert ( "gesdotol", "2A84" );
  m_namedEntities.insert ( "gesl", "FE00" );
  m_namedEntities.insert ( "gesles", "2A94" );
  m_namedEntities.insert ( "gfr", "DD24" );
  m_namedEntities.insert ( "gg", "226B" );
  m_namedEntities.insert ( "ggg", "22D9" );
  m_namedEntities.insert ( "gimel", "2137" );
  m_namedEntities.insert ( "gjcy", "0453" );
  m_namedEntities.insert ( "gl", "2277" );
  m_namedEntities.insert ( "glE", "2A92" );
  m_namedEntities.insert ( "gla", "2AA5" );
  m_namedEntities.insert ( "glj", "2AA4" );
  m_namedEntities.insert ( "gnE", "2269" );
  m_namedEntities.insert ( "gnap", "2A8A" );
  m_namedEntities.insert ( "gnapprox", "2A8A" );
  m_namedEntities.insert ( "gne", "2A88" );
  m_namedEntities.insert ( "gneq", "2A88" );
  m_namedEntities.insert ( "gneqq", "2269" );
  m_namedEntities.insert ( "gnsim", "22E7" );
  m_namedEntities.insert ( "gopf", "DD58" );
  m_namedEntities.insert ( "grave", "0060" );
  m_namedEntities.insert ( "gscr", "210A" );
  m_namedEntities.insert ( "gsim", "2273" );
  m_namedEntities.insert ( "gsime", "2A8E" );
  m_namedEntities.insert ( "gsiml", "2A90" );
  m_namedEntities.insert ( "gt", "003E" );
  m_namedEntities.insert ( "gtcc", "2AA7" );
  m_namedEntities.insert ( "gtcir", "2A7A" );
  m_namedEntities.insert ( "gtdot", "22D7" );
  m_namedEntities.insert ( "gtlPar", "2995" );
  m_namedEntities.insert ( "gtquest", "2A7C" );
  m_namedEntities.insert ( "gtrapprox", "2A86" );
  m_namedEntities.insert ( "gtrarr", "2978" );
  m_namedEntities.insert ( "gtrdot", "22D7" );
  m_namedEntities.insert ( "gtreqless", "22DB" );
  m_namedEntities.insert ( "gtreqqless", "2A8C" );
  m_namedEntities.insert ( "gtrless", "2277" );
  m_namedEntities.insert ( "gtrsim", "2273" );
  m_namedEntities.insert ( "gvertneqq", "FE00" );
  m_namedEntities.insert ( "gvnE", "FE00" );
  m_namedEntities.insert ( "hArr", "21D4" );
  m_namedEntities.insert ( "hairsp", "200A" );
  m_namedEntities.insert ( "half", "00BD" );
  m_namedEntities.insert ( "hamilt", "210B" );
  m_namedEntities.insert ( "hardcy", "044A" );
  m_namedEntities.insert ( "harr", "2194" );
  m_namedEntities.insert ( "harrcir", "2948" );
  m_namedEntities.insert ( "harrw", "21AD" );
  m_namedEntities.insert ( "hbar", "210F" );
  m_namedEntities.insert ( "hcirc", "0125" );
  m_namedEntities.insert ( "hearts", "2665" );
  m_namedEntities.insert ( "heartsuit", "2665" );
  m_namedEntities.insert ( "hellip", "2026" );
  m_namedEntities.insert ( "hercon", "22B9" );
  m_namedEntities.insert ( "hfr", "DD25" );
  m_namedEntities.insert ( "hksearow", "2925" );
  m_namedEntities.insert ( "hkswarow", "2926" );
  m_namedEntities.insert ( "hoarr", "21FF" );
  m_namedEntities.insert ( "homtht", "223B" );
  m_namedEntities.insert ( "hookleftarrow", "21A9" );
  m_namedEntities.insert ( "hookrightarrow", "21AA" );
  m_namedEntities.insert ( "hopf", "DD59" );
  m_namedEntities.insert ( "horbar", "2015" );
  m_namedEntities.insert ( "hscr", "DCBD" );
  m_namedEntities.insert ( "hslash", "210F" );
  m_namedEntities.insert ( "hstrok", "0127" );
  m_namedEntities.insert ( "hybull", "2043" );
  m_namedEntities.insert ( "hyphen", "2010" );
  m_namedEntities.insert ( "iacute", "00ED" );
  m_namedEntities.insert ( "ic", "2063" );
  m_namedEntities.insert ( "icirc", "00EE" );
  m_namedEntities.insert ( "icy", "0438" );
  m_namedEntities.insert ( "iecy", "0435" );
  m_namedEntities.insert ( "iexcl", "00A1" );
  m_namedEntities.insert ( "iff", "21D4" );
  m_namedEntities.insert ( "ifr", "DD26" );
  m_namedEntities.insert ( "igrave", "00EC" );
  m_namedEntities.insert ( "ii", "2148" );
  m_namedEntities.insert ( "iiiint", "2A0C" );
  m_namedEntities.insert ( "iiint", "222D" );
  m_namedEntities.insert ( "iinfin", "29DC" );
  m_namedEntities.insert ( "iiota", "2129" );
  m_namedEntities.insert ( "ijlig", "0133" );
  m_namedEntities.insert ( "imacr", "012B" );
  m_namedEntities.insert ( "image", "2111" );
  m_namedEntities.insert ( "imagline", "2110" );
  m_namedEntities.insert ( "imagpart", "2111" );
  m_namedEntities.insert ( "imath", "0131" );
  m_namedEntities.insert ( "imof", "22B7" );
  m_namedEntities.insert ( "imped", "01B5" );
  m_namedEntities.insert ( "in", "2208" );
  m_namedEntities.insert ( "incare", "2105" );
  m_namedEntities.insert ( "infin", "221E" );
  m_namedEntities.insert ( "infintie", "29DD" );
  m_namedEntities.insert ( "inodot", "0131" );
  m_namedEntities.insert ( "int", "222B" );
  m_namedEntities.insert ( "intcal", "22BA" );
  m_namedEntities.insert ( "integers", "2124" );
  m_namedEntities.insert ( "intercal", "22BA" );
  m_namedEntities.insert ( "intlarhk", "2A17" );
  m_namedEntities.insert ( "intprod", "2A3C" );
  m_namedEntities.insert ( "iocy", "0451" );
  m_namedEntities.insert ( "iogon", "012F" );
  m_namedEntities.insert ( "iopf", "DD5A" );
  m_namedEntities.insert ( "iota", "03B9" );
  m_namedEntities.insert ( "iprod", "2A3C" );
  m_namedEntities.insert ( "iquest", "00BF" );
  m_namedEntities.insert ( "iscr", "DCBE" );
  m_namedEntities.insert ( "isin", "2208" );
  m_namedEntities.insert ( "isinE", "22F9" );
  m_namedEntities.insert ( "isindot", "22F5" );
  m_namedEntities.insert ( "isins", "22F4" );
  m_namedEntities.insert ( "isinsv", "22F3" );
  m_namedEntities.insert ( "isinv", "2208" );
  m_namedEntities.insert ( "it", "2062" );
  m_namedEntities.insert ( "itilde", "0129" );
  m_namedEntities.insert ( "iukcy", "0456" );
  m_namedEntities.insert ( "iuml", "00EF" );
  m_namedEntities.insert ( "jcirc", "0135" );
  m_namedEntities.insert ( "jcy", "0439" );
  m_namedEntities.insert ( "jfr", "DD27" );
  m_namedEntities.insert ( "jmath", "0237" );
  m_namedEntities.insert ( "jopf", "DD5B" );
  m_namedEntities.insert ( "jscr", "DCBF" );
  m_namedEntities.insert ( "jsercy", "0458" );
  m_namedEntities.insert ( "jukcy", "0454" );
  m_namedEntities.insert ( "kappa", "03BA" );
  m_namedEntities.insert ( "kappav", "03F0" );
  m_namedEntities.insert ( "kcedil", "0137" );
  m_namedEntities.insert ( "kcy", "043A" );
  m_namedEntities.insert ( "kfr", "DD28" );
  m_namedEntities.insert ( "kgreen", "0138" );
  m_namedEntities.insert ( "khcy", "0445" );
  m_namedEntities.insert ( "kjcy", "045C" );
  m_namedEntities.insert ( "kopf", "DD5C" );
  m_namedEntities.insert ( "kscr", "DCC0" );
  m_namedEntities.insert ( "lAarr", "21DA" );
  m_namedEntities.insert ( "lArr", "21D0" );
  m_namedEntities.insert ( "lAtail", "291B" );
  m_namedEntities.insert ( "lBarr", "290E" );
  m_namedEntities.insert ( "lE", "2266" );
  m_namedEntities.insert ( "lEg", "2A8B" );
  m_namedEntities.insert ( "lHar", "2962" );
  m_namedEntities.insert ( "lacute", "013A" );
  m_namedEntities.insert ( "laemptyv", "29B4" );
  m_namedEntities.insert ( "lagran", "2112" );
  m_namedEntities.insert ( "lambda", "03BB" );
  m_namedEntities.insert ( "lang", "27E8" );
  m_namedEntities.insert ( "langd", "2991" );
  m_namedEntities.insert ( "langle", "27E8" );
  m_namedEntities.insert ( "lap", "2A85" );
  m_namedEntities.insert ( "laquo", "00AB" );
  m_namedEntities.insert ( "larr", "2190" );
  m_namedEntities.insert ( "larrb", "21E4" );
  m_namedEntities.insert ( "larrbfs", "291F" );
  m_namedEntities.insert ( "larrfs", "291D" );
  m_namedEntities.insert ( "larrhk", "21A9" );
  m_namedEntities.insert ( "larrlp", "21AB" );
  m_namedEntities.insert ( "larrpl", "2939" );
  m_namedEntities.insert ( "larrsim", "2973" );
  m_namedEntities.insert ( "larrtl", "21A2" );
  m_namedEntities.insert ( "lat", "2AAB" );
  m_namedEntities.insert ( "latail", "2919" );
  m_namedEntities.insert ( "late", "2AAD" );
  m_namedEntities.insert ( "lates", "FE00" );
  m_namedEntities.insert ( "lbarr", "290C" );
  m_namedEntities.insert ( "lbbrk", "2772" );
  m_namedEntities.insert ( "lbrace", "007B" );
  m_namedEntities.insert ( "lbrack", "005B" );
  m_namedEntities.insert ( "lbrke", "298B" );
  m_namedEntities.insert ( "lbrksld", "298F" );
  m_namedEntities.insert ( "lbrkslu", "298D" );
  m_namedEntities.insert ( "lcaron", "013E" );
  m_namedEntities.insert ( "lcedil", "013C" );
  m_namedEntities.insert ( "lceil", "2308" );
  m_namedEntities.insert ( "lcub", "007B" );
  m_namedEntities.insert ( "lcy", "043B" );
  m_namedEntities.insert ( "ldca", "2936" );
  m_namedEntities.insert ( "ldquo", "201C" );
  m_namedEntities.insert ( "ldquor", "201E" );
  m_namedEntities.insert ( "ldrdhar", "2967" );
  m_namedEntities.insert ( "ldrushar", "294B" );
  m_namedEntities.insert ( "ldsh", "21B2" );
  m_namedEntities.insert ( "le", "2264" );
  m_namedEntities.insert ( "leftarrow", "2190" );
  m_namedEntities.insert ( "leftarrowtail", "21A2" );
  m_namedEntities.insert ( "leftharpoondown", "21BD" );
  m_namedEntities.insert ( "leftharpoonup", "21BC" );
  m_namedEntities.insert ( "leftleftarrows", "21C7" );
  m_namedEntities.insert ( "leftrightarrow", "2194" );
  m_namedEntities.insert ( "leftrightarrows", "21C6" );
  m_namedEntities.insert ( "leftrightharpoons", "21CB" );
  m_namedEntities.insert ( "leftrightsquigarrow", "21AD" );
  m_namedEntities.insert ( "leftthreetimes", "22CB" );
  m_namedEntities.insert ( "leg", "22DA" );
  m_namedEntities.insert ( "leq", "2264" );
  m_namedEntities.insert ( "leqq", "2266" );
  m_namedEntities.insert ( "leqslant", "2A7D" );
  m_namedEntities.insert ( "les", "2A7D" );
  m_namedEntities.insert ( "lescc", "2AA8" );
  m_namedEntities.insert ( "lesdot", "2A7F" );
  m_namedEntities.insert ( "lesdoto", "2A81" );
  m_namedEntities.insert ( "lesdotor", "2A83" );
  m_namedEntities.insert ( "lesg", "FE00" );
  m_namedEntities.insert ( "lesges", "2A93" );
  m_namedEntities.insert ( "lessapprox", "2A85" );
  m_namedEntities.insert ( "lessdot", "22D6" );
  m_namedEntities.insert ( "lesseqgtr", "22DA" );
  m_namedEntities.insert ( "lesseqqgtr", "2A8B" );
  m_namedEntities.insert ( "lessgtr", "2276" );
  m_namedEntities.insert ( "lesssim", "2272" );
  m_namedEntities.insert ( "lfisht", "297C" );
  m_namedEntities.insert ( "lfloor", "230A" );
  m_namedEntities.insert ( "lfr", "DD29" );
  m_namedEntities.insert ( "lg", "2276" );
  m_namedEntities.insert ( "lgE", "2A91" );
  m_namedEntities.insert ( "lhard", "21BD" );
  m_namedEntities.insert ( "lharu", "21BC" );
  m_namedEntities.insert ( "lharul", "296A" );
  m_namedEntities.insert ( "lhblk", "2584" );
  m_namedEntities.insert ( "ljcy", "0459" );
  m_namedEntities.insert ( "ll", "226A" );
  m_namedEntities.insert ( "llarr", "21C7" );
  m_namedEntities.insert ( "llcorner", "231E" );
  m_namedEntities.insert ( "llhard", "296B" );
  m_namedEntities.insert ( "lltri", "25FA" );
  m_namedEntities.insert ( "lmidot", "0140" );
  m_namedEntities.insert ( "lmoust", "23B0" );
  m_namedEntities.insert ( "lmoustache", "23B0" );
  m_namedEntities.insert ( "lnE", "2268" );
  m_namedEntities.insert ( "lnap", "2A89" );
  m_namedEntities.insert ( "lnapprox", "2A89" );
  m_namedEntities.insert ( "lne", "2A87" );
  m_namedEntities.insert ( "lneq", "2A87" );
  m_namedEntities.insert ( "lneqq", "2268" );
  m_namedEntities.insert ( "lnsim", "22E6" );
  m_namedEntities.insert ( "loang", "27EC" );
  m_namedEntities.insert ( "loarr", "21FD" );
  m_namedEntities.insert ( "lobrk", "27E6" );
  m_namedEntities.insert ( "longleftarrow", "27F5" );
  m_namedEntities.insert ( "longleftrightarrow", "27F7" );
  m_namedEntities.insert ( "longmapsto", "27FC" );
  m_namedEntities.insert ( "longrightarrow", "27F6" );
  m_namedEntities.insert ( "looparrowleft", "21AB" );
  m_namedEntities.insert ( "looparrowright", "21AC" );
  m_namedEntities.insert ( "lopar", "2985" );
  m_namedEntities.insert ( "lopf", "DD5D" );
  m_namedEntities.insert ( "loplus", "2A2D" );
  m_namedEntities.insert ( "lotimes", "2A34" );
  m_namedEntities.insert ( "lowast", "2217" );
  m_namedEntities.insert ( "lowbar", "005F" );
  m_namedEntities.insert ( "loz", "25CA" );
  m_namedEntities.insert ( "lozenge", "25CA" );
  m_namedEntities.insert ( "lozf", "29EB" );
  m_namedEntities.insert ( "lpar", "0028" );
  m_namedEntities.insert ( "lparlt", "2993" );
  m_namedEntities.insert ( "lrarr", "21C6" );
  m_namedEntities.insert ( "lrcorner", "231F" );
  m_namedEntities.insert ( "lrhar", "21CB" );
  m_namedEntities.insert ( "lrhard", "296D" );
  m_namedEntities.insert ( "lrm", "200E" );
  m_namedEntities.insert ( "lrtri", "22BF" );
  m_namedEntities.insert ( "lsaquo", "2039" );
  m_namedEntities.insert ( "lscr", "DCC1" );
  m_namedEntities.insert ( "lsh", "21B0" );
  m_namedEntities.insert ( "lsim", "2272" );
  m_namedEntities.insert ( "lsime", "2A8D" );
  m_namedEntities.insert ( "lsimg", "2A8F" );
  m_namedEntities.insert ( "lsqb", "005B" );
  m_namedEntities.insert ( "lsquo", "2018" );
  m_namedEntities.insert ( "lsquor", "201A" );
  m_namedEntities.insert ( "lstrok", "0142" );
  m_namedEntities.insert ( "lt", "003C" );
  m_namedEntities.insert ( "ltcc", "2AA6" );
  m_namedEntities.insert ( "ltcir", "2A79" );
  m_namedEntities.insert ( "ltdot", "22D6" );
  m_namedEntities.insert ( "lthree", "22CB" );
  m_namedEntities.insert ( "ltimes", "22C9" );
  m_namedEntities.insert ( "ltlarr", "2976" );
  m_namedEntities.insert ( "ltquest", "2A7B" );
  m_namedEntities.insert ( "ltrPar", "2996" );
  m_namedEntities.insert ( "ltri", "25C3" );
  m_namedEntities.insert ( "ltrie", "22B4" );
  m_namedEntities.insert ( "ltrif", "25C2" );
  m_namedEntities.insert ( "lurdshar", "294A" );
  m_namedEntities.insert ( "luruhar", "2966" );
  m_namedEntities.insert ( "lvertneqq", "FE00" );
  m_namedEntities.insert ( "lvnE", "FE00" );
  m_namedEntities.insert ( "mDDot", "223A" );
  m_namedEntities.insert ( "macr", "00AF" );
  m_namedEntities.insert ( "male", "2642" );
  m_namedEntities.insert ( "malt", "2720" );
  m_namedEntities.insert ( "maltese", "2720" );
  m_namedEntities.insert ( "map", "21A6" );
  m_namedEntities.insert ( "mapsto", "21A6" );
  m_namedEntities.insert ( "mapstodown", "21A7" );
  m_namedEntities.insert ( "mapstoleft", "21A4" );
  m_namedEntities.insert ( "mapstoup", "21A5" );
  m_namedEntities.insert ( "marker", "25AE" );
  m_namedEntities.insert ( "mcomma", "2A29" );
  m_namedEntities.insert ( "mcy", "043C" );
  m_namedEntities.insert ( "mdash", "2014" );
  m_namedEntities.insert ( "measuredangle", "2221" );
  m_namedEntities.insert ( "mfr", "DD2A" );
  m_namedEntities.insert ( "mho", "2127" );
  m_namedEntities.insert ( "micro", "00B5" );
  m_namedEntities.insert ( "mid", "2223" );
  m_namedEntities.insert ( "midast", "002A" );
  m_namedEntities.insert ( "midcir", "2AF0" );
  m_namedEntities.insert ( "middot", "00B7" );
  m_namedEntities.insert ( "minus", "2212" );
  m_namedEntities.insert ( "minusb", "229F" );
  m_namedEntities.insert ( "minusd", "2238" );
  m_namedEntities.insert ( "minusdu", "2A2A" );
  m_namedEntities.insert ( "mlcp", "2ADB" );
  m_namedEntities.insert ( "mldr", "2026" );
  m_namedEntities.insert ( "mnplus", "2213" );
  m_namedEntities.insert ( "models", "22A7" );
  m_namedEntities.insert ( "mopf", "DD5E" );
  m_namedEntities.insert ( "mp", "2213" );
  m_namedEntities.insert ( "mscr", "DCC2" );
  m_namedEntities.insert ( "mstpos", "223E" );
  m_namedEntities.insert ( "mu", "03BC" );
  m_namedEntities.insert ( "multimap", "22B8" );
  m_namedEntities.insert ( "mumap", "22B8" );
  m_namedEntities.insert ( "nGg", "0338" );
  m_namedEntities.insert ( "nGt", "20D2" );
  m_namedEntities.insert ( "nGtv", "0338" );
  m_namedEntities.insert ( "nLeftarrow", "21CD" );
  m_namedEntities.insert ( "nLeftrightarrow", "21CE" );
  m_namedEntities.insert ( "nLl", "0338" );
  m_namedEntities.insert ( "nLt", "20D2" );
  m_namedEntities.insert ( "nLtv", "0338" );
  m_namedEntities.insert ( "nRightarrow", "21CF" );
  m_namedEntities.insert ( "nVDash", "22AF" );
  m_namedEntities.insert ( "nVdash", "22AE" );
  m_namedEntities.insert ( "nabla", "2207" );
  m_namedEntities.insert ( "nacute", "0144" );
  m_namedEntities.insert ( "nang", "20D2" );
  m_namedEntities.insert ( "nap", "2249" );
  m_namedEntities.insert ( "napE", "0338" );
  m_namedEntities.insert ( "napid", "0338" );
  m_namedEntities.insert ( "napos", "0149" );
  m_namedEntities.insert ( "napprox", "2249" );
  m_namedEntities.insert ( "natur", "266E" );
  m_namedEntities.insert ( "natural", "266E" );
  m_namedEntities.insert ( "naturals", "2115" );
  m_namedEntities.insert ( "nbsp", "00A0" );
  m_namedEntities.insert ( "nbump", "0338" );
  m_namedEntities.insert ( "nbumpe", "0338" );
  m_namedEntities.insert ( "ncap", "2A43" );
  m_namedEntities.insert ( "ncaron", "0148" );
  m_namedEntities.insert ( "ncedil", "0146" );
  m_namedEntities.insert ( "ncong", "2247" );
  m_namedEntities.insert ( "ncongdot", "0338" );
  m_namedEntities.insert ( "ncup", "2A42" );
  m_namedEntities.insert ( "ncy", "043D" );
  m_namedEntities.insert ( "ndash", "2013" );
  m_namedEntities.insert ( "ne", "2260" );
  m_namedEntities.insert ( "neArr", "21D7" );
  m_namedEntities.insert ( "nearhk", "2924" );
  m_namedEntities.insert ( "nearr", "2197" );
  m_namedEntities.insert ( "nearrow", "2197" );
  m_namedEntities.insert ( "nedot", "0338" );
  m_namedEntities.insert ( "nequiv", "2262" );
  m_namedEntities.insert ( "nesear", "2928" );
  m_namedEntities.insert ( "nesim", "0338" );
  m_namedEntities.insert ( "nexist", "2204" );
  m_namedEntities.insert ( "nexists", "2204" );
  m_namedEntities.insert ( "nfr", "DD2B" );
  m_namedEntities.insert ( "ngE", "0338" );
  m_namedEntities.insert ( "nge", "2271" );
  m_namedEntities.insert ( "ngeq", "2271" );
  m_namedEntities.insert ( "ngeqq", "0338" );
  m_namedEntities.insert ( "ngeqslant", "0338" );
  m_namedEntities.insert ( "nges", "0338" );
  m_namedEntities.insert ( "ngsim", "2275" );
  m_namedEntities.insert ( "ngt", "226F" );
  m_namedEntities.insert ( "ngtr", "226F" );
  m_namedEntities.insert ( "nhArr", "21CE" );
  m_namedEntities.insert ( "nharr", "21AE" );
  m_namedEntities.insert ( "nhpar", "2AF2" );
  m_namedEntities.insert ( "ni", "220B" );
  m_namedEntities.insert ( "nis", "22FC" );
  m_namedEntities.insert ( "nisd", "22FA" );
  m_namedEntities.insert ( "niv", "220B" );
  m_namedEntities.insert ( "njcy", "045A" );
  m_namedEntities.insert ( "nlArr", "21CD" );
  m_namedEntities.insert ( "nlE", "0338" );
  m_namedEntities.insert ( "nlarr", "219A" );
  m_namedEntities.insert ( "nldr", "2025" );
  m_namedEntities.insert ( "nle", "2270" );
  m_namedEntities.insert ( "nleftarrow", "219A" );
  m_namedEntities.insert ( "nleftrightarrow", "21AE" );
  m_namedEntities.insert ( "nleq", "2270" );
  m_namedEntities.insert ( "nleqq", "0338" );
  m_namedEntities.insert ( "nleqslant", "0338" );
  m_namedEntities.insert ( "nles", "0338" );
  m_namedEntities.insert ( "nless", "226E" );
  m_namedEntities.insert ( "nlsim", "2274" );
  m_namedEntities.insert ( "nlt", "226E" );
  m_namedEntities.insert ( "nltri", "22EA" );
  m_namedEntities.insert ( "nltrie", "22EC" );
  m_namedEntities.insert ( "nmid", "2224" );
  m_namedEntities.insert ( "nopf", "DD5F" );
  m_namedEntities.insert ( "not", "00AC" );
  m_namedEntities.insert ( "notin", "2209" );
  m_namedEntities.insert ( "notinE", "0338" );
  m_namedEntities.insert ( "notindot", "0338" );
  m_namedEntities.insert ( "notinva", "2209" );
  m_namedEntities.insert ( "notinvb", "22F7" );
  m_namedEntities.insert ( "notinvc", "22F6" );
  m_namedEntities.insert ( "notni", "220C" );
  m_namedEntities.insert ( "notniva", "220C" );
  m_namedEntities.insert ( "notnivb", "22FE" );
  m_namedEntities.insert ( "notnivc", "22FD" );
  m_namedEntities.insert ( "npar", "2226" );
  m_namedEntities.insert ( "nparallel", "2226" );
  m_namedEntities.insert ( "nparsl", "20E5" );
  m_namedEntities.insert ( "npart", "0338" );
  m_namedEntities.insert ( "npolint", "2A14" );
  m_namedEntities.insert ( "npr", "2280" );
  m_namedEntities.insert ( "nprcue", "22E0" );
  m_namedEntities.insert ( "npre", "0338" );
  m_namedEntities.insert ( "nprec", "2280" );
  m_namedEntities.insert ( "npreceq", "0338" );
  m_namedEntities.insert ( "nrArr", "21CF" );
  m_namedEntities.insert ( "nrarr", "219B" );
  m_namedEntities.insert ( "nrarrc", "0338" );
  m_namedEntities.insert ( "nrarrw", "0338" );
  m_namedEntities.insert ( "nrightarrow", "219B" );
  m_namedEntities.insert ( "nrtri", "22EB" );
  m_namedEntities.insert ( "nrtrie", "22ED" );
  m_namedEntities.insert ( "nsc", "2281" );
  m_namedEntities.insert ( "nsccue", "22E1" );
  m_namedEntities.insert ( "nsce", "0338" );
  m_namedEntities.insert ( "nscr", "DCC3" );
  m_namedEntities.insert ( "nshortmid", "2224" );
  m_namedEntities.insert ( "nshortparallel", "2226" );
  m_namedEntities.insert ( "nsim", "2241" );
  m_namedEntities.insert ( "nsime", "2244" );
  m_namedEntities.insert ( "nsimeq", "2244" );
  m_namedEntities.insert ( "nsmid", "2224" );
  m_namedEntities.insert ( "nspar", "2226" );
  m_namedEntities.insert ( "nsqsube", "22E2" );
  m_namedEntities.insert ( "nsqsupe", "22E3" );
  m_namedEntities.insert ( "nsub", "2284" );
  m_namedEntities.insert ( "nsubE", "0338" );
  m_namedEntities.insert ( "nsube", "2288" );
  m_namedEntities.insert ( "nsubset", "20D2" );
  m_namedEntities.insert ( "nsubseteq", "2288" );
  m_namedEntities.insert ( "nsubseteqq", "0338" );
  m_namedEntities.insert ( "nsucc", "2281" );
  m_namedEntities.insert ( "nsucceq", "0338" );
  m_namedEntities.insert ( "nsup", "2285" );
  m_namedEntities.insert ( "nsupE", "0338" );
  m_namedEntities.insert ( "nsupe", "2289" );
  m_namedEntities.insert ( "nsupset", "20D2" );
  m_namedEntities.insert ( "nsupseteq", "2289" );
  m_namedEntities.insert ( "nsupseteqq", "0338" );
  m_namedEntities.insert ( "ntgl", "2279" );
  m_namedEntities.insert ( "ntilde", "00F1" );
  m_namedEntities.insert ( "ntlg", "2278" );
  m_namedEntities.insert ( "ntriangleleft", "22EA" );
  m_namedEntities.insert ( "ntrianglelefteq", "22EC" );
  m_namedEntities.insert ( "ntriangleright", "22EB" );
  m_namedEntities.insert ( "ntrianglerighteq", "22ED" );
  m_namedEntities.insert ( "nu", "03BD" );
  m_namedEntities.insert ( "num", "0023" );
  m_namedEntities.insert ( "numero", "2116" );
  m_namedEntities.insert ( "numsp", "2007" );
  m_namedEntities.insert ( "nvDash", "22AD" );
  m_namedEntities.insert ( "nvHarr", "2904" );
  m_namedEntities.insert ( "nvap", "20D2" );
  m_namedEntities.insert ( "nvdash", "22AC" );
  m_namedEntities.insert ( "nvge", "20D2" );
  m_namedEntities.insert ( "nvgt", "20D2" );
  m_namedEntities.insert ( "nvinfin", "29DE" );
  m_namedEntities.insert ( "nvlArr", "2902" );
  m_namedEntities.insert ( "nvle", "20D2" );
  m_namedEntities.insert ( "nvlt", "20D2" );
  m_namedEntities.insert ( "nvltrie", "20D2" );
  m_namedEntities.insert ( "nvrArr", "2903" );
  m_namedEntities.insert ( "nvrtrie", "20D2" );
  m_namedEntities.insert ( "nvsim", "20D2" );
  m_namedEntities.insert ( "nwArr", "21D6" );
  m_namedEntities.insert ( "nwarhk", "2923" );
  m_namedEntities.insert ( "nwarr", "2196" );
  m_namedEntities.insert ( "nwarrow", "2196" );
  m_namedEntities.insert ( "nwnear", "2927" );
  m_namedEntities.insert ( "oS", "24C8" );
  m_namedEntities.insert ( "oacute", "00F3" );
  m_namedEntities.insert ( "oast", "229B" );
  m_namedEntities.insert ( "ocir", "229A" );
  m_namedEntities.insert ( "ocirc", "00F4" );
  m_namedEntities.insert ( "ocy", "043E" );
  m_namedEntities.insert ( "odash", "229D" );
  m_namedEntities.insert ( "odblac", "0151" );
  m_namedEntities.insert ( "odiv", "2A38" );
  m_namedEntities.insert ( "odot", "2299" );
  m_namedEntities.insert ( "odsold", "29BC" );
  m_namedEntities.insert ( "oelig", "0153" );
  m_namedEntities.insert ( "ofcir", "29BF" );
  m_namedEntities.insert ( "ofr", "DD2C" );
  m_namedEntities.insert ( "ogon", "02DB" );
  m_namedEntities.insert ( "ograve", "00F2" );
  m_namedEntities.insert ( "ogt", "29C1" );
  m_namedEntities.insert ( "ohbar", "29B5" );
  m_namedEntities.insert ( "ohm", "03A9" );
  m_namedEntities.insert ( "oint", "222E" );
  m_namedEntities.insert ( "olarr", "21BA" );
  m_namedEntities.insert ( "olcir", "29BE" );
  m_namedEntities.insert ( "olcross", "29BB" );
  m_namedEntities.insert ( "oline", "203E" );
  m_namedEntities.insert ( "olt", "29C0" );
  m_namedEntities.insert ( "omacr", "014D" );
  m_namedEntities.insert ( "omega", "03C9" );
  m_namedEntities.insert ( "omicron", "03BF" );
  m_namedEntities.insert ( "omid", "29B6" );
  m_namedEntities.insert ( "ominus", "2296" );
  m_namedEntities.insert ( "oopf", "DD60" );
  m_namedEntities.insert ( "opar", "29B7" );
  m_namedEntities.insert ( "operp", "29B9" );
  m_namedEntities.insert ( "oplus", "2295" );
  m_namedEntities.insert ( "or", "2228" );
  m_namedEntities.insert ( "orarr", "21BB" );
  m_namedEntities.insert ( "ord", "2A5D" );
  m_namedEntities.insert ( "order", "2134" );
  m_namedEntities.insert ( "orderof", "2134" );
  m_namedEntities.insert ( "ordf", "00AA" );
  m_namedEntities.insert ( "ordm", "00BA" );
  m_namedEntities.insert ( "origof", "22B6" );
  m_namedEntities.insert ( "oror", "2A56" );
  m_namedEntities.insert ( "orslope", "2A57" );
  m_namedEntities.insert ( "orv", "2A5B" );
  m_namedEntities.insert ( "oscr", "2134" );
  m_namedEntities.insert ( "oslash", "00F8" );
  m_namedEntities.insert ( "osol", "2298" );
  m_namedEntities.insert ( "otilde", "00F5" );
  m_namedEntities.insert ( "otimes", "2297" );
  m_namedEntities.insert ( "otimesas", "2A36" );
  m_namedEntities.insert ( "ouml", "00F6" );
  m_namedEntities.insert ( "ovbar", "233D" );
  m_namedEntities.insert ( "par", "2225" );
  m_namedEntities.insert ( "para", "00B6" );
  m_namedEntities.insert ( "parallel", "2225" );
  m_namedEntities.insert ( "parsim", "2AF3" );
  m_namedEntities.insert ( "parsl", "2AFD" );
  m_namedEntities.insert ( "part", "2202" );
  m_namedEntities.insert ( "pcy", "043F" );
  m_namedEntities.insert ( "percnt", "0025" );
  m_namedEntities.insert ( "period", "002E" );
  m_namedEntities.insert ( "permil", "2030" );
  m_namedEntities.insert ( "perp", "22A5" );
  m_namedEntities.insert ( "pertenk", "2031" );
  m_namedEntities.insert ( "pfr", "DD2D" );
  m_namedEntities.insert ( "phi", "03C6" );
  m_namedEntities.insert ( "phiv", "03D5" );
  m_namedEntities.insert ( "phmmat", "2133" );
  m_namedEntities.insert ( "phone", "260E" );
  m_namedEntities.insert ( "pi", "03C0" );
  m_namedEntities.insert ( "pitchfork", "22D4" );
  m_namedEntities.insert ( "piv", "03D6" );
  m_namedEntities.insert ( "planck", "210F" );
  m_namedEntities.insert ( "planckh", "210E" );
  m_namedEntities.insert ( "plankv", "210F" );
  m_namedEntities.insert ( "plus", "002B" );
  m_namedEntities.insert ( "plusacir", "2A23" );
  m_namedEntities.insert ( "plusb", "229E" );
  m_namedEntities.insert ( "pluscir", "2A22" );
  m_namedEntities.insert ( "plusdo", "2214" );
  m_namedEntities.insert ( "plusdu", "2A25" );
  m_namedEntities.insert ( "pluse", "2A72" );
  m_namedEntities.insert ( "plusmn", "00B1" );
  m_namedEntities.insert ( "plussim", "2A26" );
  m_namedEntities.insert ( "plustwo", "2A27" );
  m_namedEntities.insert ( "pm", "00B1" );
  m_namedEntities.insert ( "pointint", "2A15" );
  m_namedEntities.insert ( "popf", "DD61" );
  m_namedEntities.insert ( "pound", "00A3" );
  m_namedEntities.insert ( "pr", "227A" );
  m_namedEntities.insert ( "prE", "2AB3" );
  m_namedEntities.insert ( "prap", "2AB7" );
  m_namedEntities.insert ( "prcue", "227C" );
  m_namedEntities.insert ( "pre", "2AAF" );
  m_namedEntities.insert ( "prec", "227A" );
  m_namedEntities.insert ( "precapprox", "2AB7" );
  m_namedEntities.insert ( "preccurlyeq", "227C" );
  m_namedEntities.insert ( "preceq", "2AAF" );
  m_namedEntities.insert ( "precnapprox", "2AB9" );
  m_namedEntities.insert ( "precneqq", "2AB5" );
  m_namedEntities.insert ( "precnsim", "22E8" );
  m_namedEntities.insert ( "precsim", "227E" );
  m_namedEntities.insert ( "prime", "2032" );
  m_namedEntities.insert ( "primes", "2119" );
  m_namedEntities.insert ( "prnE", "2AB5" );
  m_namedEntities.insert ( "prnap", "2AB9" );
  m_namedEntities.insert ( "prnsim", "22E8" );
  m_namedEntities.insert ( "prod", "220F" );
  m_namedEntities.insert ( "profalar", "232E" );
  m_namedEntities.insert ( "profline", "2312" );
  m_namedEntities.insert ( "profsurf", "2313" );
  m_namedEntities.insert ( "prop", "221D" );
  m_namedEntities.insert ( "propto", "221D" );
  m_namedEntities.insert ( "prsim", "227E" );
  m_namedEntities.insert ( "prurel", "22B0" );
  m_namedEntities.insert ( "pscr", "DCC5" );
  m_namedEntities.insert ( "psi", "03C8" );
  m_namedEntities.insert ( "puncsp", "2008" );
  m_namedEntities.insert ( "qfr", "DD2E" );
  m_namedEntities.insert ( "qint", "2A0C" );
  m_namedEntities.insert ( "qopf", "DD62" );
  m_namedEntities.insert ( "qprime", "2057" );
  m_namedEntities.insert ( "qscr", "DCC6" );
  m_namedEntities.insert ( "quaternions", "210D" );
  m_namedEntities.insert ( "quatint", "2A16" );
  m_namedEntities.insert ( "quest", "003F" );
  m_namedEntities.insert ( "questeq", "225F" );
  m_namedEntities.insert ( "quot", "0022" );
  m_namedEntities.insert ( "rAarr", "21DB" );
  m_namedEntities.insert ( "rArr", "21D2" );
  m_namedEntities.insert ( "rAtail", "291C" );
  m_namedEntities.insert ( "rBarr", "290F" );
  m_namedEntities.insert ( "rHar", "2964" );
  m_namedEntities.insert ( "race", "0331" );
  m_namedEntities.insert ( "racute", "0155" );
  m_namedEntities.insert ( "radic", "221A" );
  m_namedEntities.insert ( "raemptyv", "29B3" );
  m_namedEntities.insert ( "rang", "27E9" );
  m_namedEntities.insert ( "rangd", "2992" );
  m_namedEntities.insert ( "range", "29A5" );
  m_namedEntities.insert ( "rangle", "27E9" );
  m_namedEntities.insert ( "raquo", "00BB" );
  m_namedEntities.insert ( "rarr", "2192" );
  m_namedEntities.insert ( "rarrap", "2975" );
  m_namedEntities.insert ( "rarrb", "21E5" );
  m_namedEntities.insert ( "rarrbfs", "2920" );
  m_namedEntities.insert ( "rarrc", "2933" );
  m_namedEntities.insert ( "rarrfs", "291E" );
  m_namedEntities.insert ( "rarrhk", "21AA" );
  m_namedEntities.insert ( "rarrlp", "21AC" );
  m_namedEntities.insert ( "rarrpl", "2945" );
  m_namedEntities.insert ( "rarrsim", "2974" );
  m_namedEntities.insert ( "rarrtl", "21A3" );
  m_namedEntities.insert ( "rarrw", "219D" );
  m_namedEntities.insert ( "ratail", "291A" );
  m_namedEntities.insert ( "ratio", "2236" );
  m_namedEntities.insert ( "rationals", "211A" );
  m_namedEntities.insert ( "rbarr", "290D" );
  m_namedEntities.insert ( "rbbrk", "2773" );
  m_namedEntities.insert ( "rbrace", "007D" );
  m_namedEntities.insert ( "rbrack", "005D" );
  m_namedEntities.insert ( "rbrke", "298C" );
  m_namedEntities.insert ( "rbrksld", "298E" );
  m_namedEntities.insert ( "rbrkslu", "2990" );
  m_namedEntities.insert ( "rcaron", "0159" );
  m_namedEntities.insert ( "rcedil", "0157" );
  m_namedEntities.insert ( "rceil", "2309" );
  m_namedEntities.insert ( "rcub", "007D" );
  m_namedEntities.insert ( "rcy", "0440" );
  m_namedEntities.insert ( "rdca", "2937" );
  m_namedEntities.insert ( "rdldhar", "2969" );
  m_namedEntities.insert ( "rdquo", "201D" );
  m_namedEntities.insert ( "rdquor", "201D" );
  m_namedEntities.insert ( "rdsh", "21B3" );
  m_namedEntities.insert ( "real", "211C" );
  m_namedEntities.insert ( "realine", "211B" );
  m_namedEntities.insert ( "realpart", "211C" );
  m_namedEntities.insert ( "reals", "211D" );
  m_namedEntities.insert ( "rect", "25AD" );
  m_namedEntities.insert ( "reg", "00AE" );
  m_namedEntities.insert ( "rfisht", "297D" );
  m_namedEntities.insert ( "rfloor", "230B" );
  m_namedEntities.insert ( "rfr", "DD2F" );
  m_namedEntities.insert ( "rhard", "21C1" );
  m_namedEntities.insert ( "rharu", "21C0" );
  m_namedEntities.insert ( "rharul", "296C" );
  m_namedEntities.insert ( "rho", "03C1" );
  m_namedEntities.insert ( "rhov", "03F1" );
  m_namedEntities.insert ( "rightarrow", "2192" );
  m_namedEntities.insert ( "rightarrowtail", "21A3" );
  m_namedEntities.insert ( "rightharpoondown", "21C1" );
  m_namedEntities.insert ( "rightharpoonup", "21C0" );
  m_namedEntities.insert ( "rightleftarrows", "21C4" );
  m_namedEntities.insert ( "rightleftharpoons", "21CC" );
  m_namedEntities.insert ( "rightrightarrows", "21C9" );
  m_namedEntities.insert ( "rightsquigarrow", "219D" );
  m_namedEntities.insert ( "rightthreetimes", "22CC" );
  m_namedEntities.insert ( "ring", "02DA" );
  m_namedEntities.insert ( "risingdotseq", "2253" );
  m_namedEntities.insert ( "rlarr", "21C4" );
  m_namedEntities.insert ( "rlhar", "21CC" );
  m_namedEntities.insert ( "rlm", "200F" );
  m_namedEntities.insert ( "rmoust", "23B1" );
  m_namedEntities.insert ( "rmoustache", "23B1" );
  m_namedEntities.insert ( "rnmid", "2AEE" );
  m_namedEntities.insert ( "roang", "27ED" );
  m_namedEntities.insert ( "roarr", "21FE" );
  m_namedEntities.insert ( "robrk", "27E7" );
  m_namedEntities.insert ( "ropar", "2986" );
  m_namedEntities.insert ( "ropf", "DD63" );
  m_namedEntities.insert ( "roplus", "2A2E" );
  m_namedEntities.insert ( "rotimes", "2A35" );
  m_namedEntities.insert ( "rpar", "0029" );
  m_namedEntities.insert ( "rpargt", "2994" );
  m_namedEntities.insert ( "rppolint", "2A12" );
  m_namedEntities.insert ( "rrarr", "21C9" );
  m_namedEntities.insert ( "rsaquo", "203A" );
  m_namedEntities.insert ( "rscr", "DCC7" );
  m_namedEntities.insert ( "rsh", "21B1" );
  m_namedEntities.insert ( "rsqb", "005D" );
  m_namedEntities.insert ( "rsquo", "2019" );
  m_namedEntities.insert ( "rsquor", "2019" );
  m_namedEntities.insert ( "rthree", "22CC" );
  m_namedEntities.insert ( "rtimes", "22CA" );
  m_namedEntities.insert ( "rtri", "25B9" );
  m_namedEntities.insert ( "rtrie", "22B5" );
  m_namedEntities.insert ( "rtrif", "25B8" );
  m_namedEntities.insert ( "rtriltri", "29CE" );
  m_namedEntities.insert ( "ruluhar", "2968" );
  m_namedEntities.insert ( "rx", "211E" );
  m_namedEntities.insert ( "sacute", "015B" );
  m_namedEntities.insert ( "sbquo", "201A" );
  m_namedEntities.insert ( "sc", "227B" );
  m_namedEntities.insert ( "scE", "2AB4" );
  m_namedEntities.insert ( "scap", "2AB8" );
  m_namedEntities.insert ( "scaron", "0161" );
  m_namedEntities.insert ( "sccue", "227D" );
  m_namedEntities.insert ( "sce", "2AB0" );
  m_namedEntities.insert ( "scedil", "015F" );
  m_namedEntities.insert ( "scirc", "015D" );
  m_namedEntities.insert ( "scnE", "2AB6" );
  m_namedEntities.insert ( "scnap", "2ABA" );
  m_namedEntities.insert ( "scnsim", "22E9" );
  m_namedEntities.insert ( "scpolint", "2A13" );
  m_namedEntities.insert ( "scsim", "227F" );
  m_namedEntities.insert ( "scy", "0441" );
  m_namedEntities.insert ( "sdot", "22C5" );
  m_namedEntities.insert ( "sdotb", "22A1" );
  m_namedEntities.insert ( "sdote", "2A66" );
  m_namedEntities.insert ( "seArr", "21D8" );
  m_namedEntities.insert ( "searhk", "2925" );
  m_namedEntities.insert ( "searr", "2198" );
  m_namedEntities.insert ( "searrow", "2198" );
  m_namedEntities.insert ( "sect", "00A7" );
  m_namedEntities.insert ( "semi", "003B" );
  m_namedEntities.insert ( "seswar", "2929" );
  m_namedEntities.insert ( "setminus", "2216" );
  m_namedEntities.insert ( "setmn", "2216" );
  m_namedEntities.insert ( "sext", "2736" );
  m_namedEntities.insert ( "sfr", "DD30" );
  m_namedEntities.insert ( "sfrown", "2322" );
  m_namedEntities.insert ( "sharp", "266F" );
  m_namedEntities.insert ( "shchcy", "0449" );
  m_namedEntities.insert ( "shcy", "0448" );
  m_namedEntities.insert ( "shortmid", "2223" );
  m_namedEntities.insert ( "shortparallel", "2225" );
  m_namedEntities.insert ( "shy", "00AD" );
  m_namedEntities.insert ( "sigma", "03C3" );
  m_namedEntities.insert ( "sigmaf", "03C2" );
  m_namedEntities.insert ( "sigmav", "03C2" );
  m_namedEntities.insert ( "sim", "223C" );
  m_namedEntities.insert ( "simdot", "2A6A" );
  m_namedEntities.insert ( "sime", "2243" );
  m_namedEntities.insert ( "simeq", "2243" );
  m_namedEntities.insert ( "simg", "2A9E" );
  m_namedEntities.insert ( "simgE", "2AA0" );
  m_namedEntities.insert ( "siml", "2A9D" );
  m_namedEntities.insert ( "simlE", "2A9F" );
  m_namedEntities.insert ( "simne", "2246" );
  m_namedEntities.insert ( "simplus", "2A24" );
  m_namedEntities.insert ( "simrarr", "2972" );
  m_namedEntities.insert ( "slarr", "2190" );
  m_namedEntities.insert ( "smallsetminus", "2216" );
  m_namedEntities.insert ( "smashp", "2A33" );
  m_namedEntities.insert ( "smeparsl", "29E4" );
  m_namedEntities.insert ( "smid", "2223" );
  m_namedEntities.insert ( "smile", "2323" );
  m_namedEntities.insert ( "smt", "2AAA" );
  m_namedEntities.insert ( "smte", "2AAC" );
  m_namedEntities.insert ( "smtes", "FE00" );
  m_namedEntities.insert ( "softcy", "044C" );
  m_namedEntities.insert ( "sol", "002F" );
  m_namedEntities.insert ( "solb", "29C4" );
  m_namedEntities.insert ( "solbar", "233F" );
  m_namedEntities.insert ( "sopf", "DD64" );
  m_namedEntities.insert ( "spades", "2660" );
  m_namedEntities.insert ( "spadesuit", "2660" );
  m_namedEntities.insert ( "spar", "2225" );
  m_namedEntities.insert ( "sqcap", "2293" );
  m_namedEntities.insert ( "sqcaps", "FE00" );
  m_namedEntities.insert ( "sqcup", "2294" );
  m_namedEntities.insert ( "sqcups", "FE00" );
  m_namedEntities.insert ( "sqsub", "228F" );
  m_namedEntities.insert ( "sqsube", "2291" );
  m_namedEntities.insert ( "sqsubset", "228F" );
  m_namedEntities.insert ( "sqsubseteq", "2291" );
  m_namedEntities.insert ( "sqsup", "2290" );
  m_namedEntities.insert ( "sqsupe", "2292" );
  m_namedEntities.insert ( "sqsupset", "2290" );
  m_namedEntities.insert ( "sqsupseteq", "2292" );
  m_namedEntities.insert ( "squ", "25A1" );
  m_namedEntities.insert ( "square", "25A1" );
  m_namedEntities.insert ( "squarf", "25AA" );
  m_namedEntities.insert ( "squf", "25AA" );
  m_namedEntities.insert ( "srarr", "2192" );
  m_namedEntities.insert ( "sscr", "DCC8" );
  m_namedEntities.insert ( "ssetmn", "2216" );
  m_namedEntities.insert ( "ssmile", "2323" );
  m_namedEntities.insert ( "sstarf", "22C6" );
  m_namedEntities.insert ( "star", "2606" );
  m_namedEntities.insert ( "starf", "2605" );
  m_namedEntities.insert ( "straightepsilon", "03F5" );
  m_namedEntities.insert ( "straightphi", "03D5" );
  m_namedEntities.insert ( "strns", "00AF" );
  m_namedEntities.insert ( "sub", "2282" );
  m_namedEntities.insert ( "subE", "2AC5" );
  m_namedEntities.insert ( "subdot", "2ABD" );
  m_namedEntities.insert ( "sube", "2286" );
  m_namedEntities.insert ( "subedot", "2AC3" );
  m_namedEntities.insert ( "submult", "2AC1" );
  m_namedEntities.insert ( "subnE", "2ACB" );
  m_namedEntities.insert ( "subne", "228A" );
  m_namedEntities.insert ( "subplus", "2ABF" );
  m_namedEntities.insert ( "subrarr", "2979" );
  m_namedEntities.insert ( "subset", "2282" );
  m_namedEntities.insert ( "subseteq", "2286" );
  m_namedEntities.insert ( "subseteqq", "2AC5" );
  m_namedEntities.insert ( "subsetneq", "228A" );
  m_namedEntities.insert ( "subsetneqq", "2ACB" );
  m_namedEntities.insert ( "subsim", "2AC7" );
  m_namedEntities.insert ( "subsub", "2AD5" );
  m_namedEntities.insert ( "subsup", "2AD3" );
  m_namedEntities.insert ( "succ", "227B" );
  m_namedEntities.insert ( "succapprox", "2AB8" );
  m_namedEntities.insert ( "succcurlyeq", "227D" );
  m_namedEntities.insert ( "succeq", "2AB0" );
  m_namedEntities.insert ( "succnapprox", "2ABA" );
  m_namedEntities.insert ( "succneqq", "2AB6" );
  m_namedEntities.insert ( "succnsim", "22E9" );
  m_namedEntities.insert ( "succsim", "227F" );
  m_namedEntities.insert ( "sum", "2211" );
  m_namedEntities.insert ( "sung", "266A" );
  m_namedEntities.insert ( "sup", "2283" );
  m_namedEntities.insert ( "supE", "2AC6" );
  m_namedEntities.insert ( "supdot", "2ABE" );
  m_namedEntities.insert ( "supdsub", "2AD8" );
  m_namedEntities.insert ( "supe", "2287" );
  m_namedEntities.insert ( "supedot", "2AC4" );
  m_namedEntities.insert ( "suphsol", "27C9" );
  m_namedEntities.insert ( "suphsub", "2AD7" );
  m_namedEntities.insert ( "suplarr", "297B" );
  m_namedEntities.insert ( "supmult", "2AC2" );
  m_namedEntities.insert ( "supnE", "2ACC" );
  m_namedEntities.insert ( "supne", "228B" );
  m_namedEntities.insert ( "supplus", "2AC0" );
  m_namedEntities.insert ( "supset", "2283" );
  m_namedEntities.insert ( "supseteq", "2287" );
  m_namedEntities.insert ( "supseteqq", "2AC6" );
  m_namedEntities.insert ( "supsetneq", "228B" );
  m_namedEntities.insert ( "supsetneqq", "2ACC" );
  m_namedEntities.insert ( "supsim", "2AC8" );
  m_namedEntities.insert ( "supsub", "2AD4" );
  m_namedEntities.insert ( "supsup", "2AD6" );
  m_namedEntities.insert ( "swArr", "21D9" );
  m_namedEntities.insert ( "swarhk", "2926" );
  m_namedEntities.insert ( "swarr", "2199" );
  m_namedEntities.insert ( "swarrow", "2199" );
  m_namedEntities.insert ( "swnwar", "292A" );
  m_namedEntities.insert ( "szlig", "00DF" );
  m_namedEntities.insert ( "target", "2316" );
  m_namedEntities.insert ( "tau", "03C4" );
  m_namedEntities.insert ( "tbrk", "23B4" );
  m_namedEntities.insert ( "tcaron", "0165" );
  m_namedEntities.insert ( "tcedil", "0163" );
  m_namedEntities.insert ( "tcy", "0442" );
  m_namedEntities.insert ( "tdot", "20DB" );
  m_namedEntities.insert ( "telrec", "2315" );
  m_namedEntities.insert ( "tfr", "DD31" );
  m_namedEntities.insert ( "therefore", "2234" );
  m_namedEntities.insert ( "theta", "03B8" );
  m_namedEntities.insert ( "thetasym", "03D1" );
  m_namedEntities.insert ( "thetav", "03D1" );
  m_namedEntities.insert ( "thickapprox", "2248" );
  m_namedEntities.insert ( "thicksim", "223C" );
  m_namedEntities.insert ( "thinsp", "2009" );
  m_namedEntities.insert ( "thkap", "2248" );
  m_namedEntities.insert ( "thksim", "223C" );
  m_namedEntities.insert ( "thorn", "00FE" );
  m_namedEntities.insert ( "tilde", "02DC" );
  m_namedEntities.insert ( "times", "00D7" );
  m_namedEntities.insert ( "timesb", "22A0" );
  m_namedEntities.insert ( "timesbar", "2A31" );
  m_namedEntities.insert ( "timesd", "2A30" );
  m_namedEntities.insert ( "tint", "222D" );
  m_namedEntities.insert ( "toea", "2928" );
  m_namedEntities.insert ( "top", "22A4" );
  m_namedEntities.insert ( "topbot", "2336" );
  m_namedEntities.insert ( "topcir", "2AF1" );
  m_namedEntities.insert ( "topf", "DD65" );
  m_namedEntities.insert ( "topfork", "2ADA" );
  m_namedEntities.insert ( "tosa", "2929" );
  m_namedEntities.insert ( "tprime", "2034" );
  m_namedEntities.insert ( "trade", "2122" );
  m_namedEntities.insert ( "triangle", "25B5" );
  m_namedEntities.insert ( "triangledown", "25BF" );
  m_namedEntities.insert ( "triangleleft", "25C3" );
  m_namedEntities.insert ( "trianglelefteq", "22B4" );
  m_namedEntities.insert ( "triangleq", "225C" );
  m_namedEntities.insert ( "triangleright", "25B9" );
  m_namedEntities.insert ( "trianglerighteq", "22B5" );
  m_namedEntities.insert ( "tridot", "25EC" );
  m_namedEntities.insert ( "trie", "225C" );
  m_namedEntities.insert ( "triminus", "2A3A" );
  m_namedEntities.insert ( "triplus", "2A39" );
  m_namedEntities.insert ( "trisb", "29CD" );
  m_namedEntities.insert ( "tritime", "2A3B" );
  m_namedEntities.insert ( "trpezium", "23E2" );
  m_namedEntities.insert ( "tscr", "DCC9" );
  m_namedEntities.insert ( "tscy", "0446" );
  m_namedEntities.insert ( "tshcy", "045B" );
  m_namedEntities.insert ( "tstrok", "0167" );
  m_namedEntities.insert ( "twixt", "226C" );
  m_namedEntities.insert ( "twoheadleftarrow", "219E" );
  m_namedEntities.insert ( "twoheadrightarrow", "21A0" );
  m_namedEntities.insert ( "uArr", "21D1" );
  m_namedEntities.insert ( "uHar", "2963" );
  m_namedEntities.insert ( "uacute", "00FA" );
  m_namedEntities.insert ( "uarr", "2191" );
  m_namedEntities.insert ( "ubrcy", "045E" );
  m_namedEntities.insert ( "ubreve", "016D" );
  m_namedEntities.insert ( "ucirc", "00FB" );
  m_namedEntities.insert ( "ucy", "0443" );
  m_namedEntities.insert ( "udarr", "21C5" );
  m_namedEntities.insert ( "udblac", "0171" );
  m_namedEntities.insert ( "udhar", "296E" );
  m_namedEntities.insert ( "ufisht", "297E" );
  m_namedEntities.insert ( "ufr", "DD32" );
  m_namedEntities.insert ( "ugrave", "00F9" );
  m_namedEntities.insert ( "uharl", "21BF" );
  m_namedEntities.insert ( "uharr", "21BE" );
  m_namedEntities.insert ( "uhblk", "2580" );
  m_namedEntities.insert ( "ulcorn", "231C" );
  m_namedEntities.insert ( "ulcorner", "231C" );
  m_namedEntities.insert ( "ulcrop", "230F" );
  m_namedEntities.insert ( "ultri", "25F8" );
  m_namedEntities.insert ( "umacr", "016B" );
  m_namedEntities.insert ( "uml", "00A8" );
  m_namedEntities.insert ( "uogon", "0173" );
  m_namedEntities.insert ( "uopf", "DD66" );
  m_namedEntities.insert ( "uparrow", "2191" );
  m_namedEntities.insert ( "updownarrow", "2195" );
  m_namedEntities.insert ( "upharpoonleft", "21BF" );
  m_namedEntities.insert ( "upharpoonright", "21BE" );
  m_namedEntities.insert ( "uplus", "228E" );
  m_namedEntities.insert ( "upsi", "03C5" );
  m_namedEntities.insert ( "upsih", "03D2" );
  m_namedEntities.insert ( "upsilon", "03C5" );
  m_namedEntities.insert ( "upuparrows", "21C8" );
  m_namedEntities.insert ( "urcorn", "231D" );
  m_namedEntities.insert ( "urcorner", "231D" );
  m_namedEntities.insert ( "urcrop", "230E" );
  m_namedEntities.insert ( "uring", "016F" );
  m_namedEntities.insert ( "urtri", "25F9" );
  m_namedEntities.insert ( "uscr", "DCCA" );
  m_namedEntities.insert ( "utdot", "22F0" );
  m_namedEntities.insert ( "utilde", "0169" );
  m_namedEntities.insert ( "utri", "25B5" );
  m_namedEntities.insert ( "utrif", "25B4" );
  m_namedEntities.insert ( "uuarr", "21C8" );
  m_namedEntities.insert ( "uuml", "00FC" );
  m_namedEntities.insert ( "uwangle", "29A7" );
  m_namedEntities.insert ( "vArr", "21D5" );
  m_namedEntities.insert ( "vBar", "2AE8" );
  m_namedEntities.insert ( "vBarv", "2AE9" );
  m_namedEntities.insert ( "vDash", "22A8" );
  m_namedEntities.insert ( "vangrt", "299C" );
  m_namedEntities.insert ( "varepsilon", "03F5" );
  m_namedEntities.insert ( "varkappa", "03F0" );
  m_namedEntities.insert ( "varnothing", "2205" );
  m_namedEntities.insert ( "varphi", "03D5" );
  m_namedEntities.insert ( "varpi", "03D6" );
  m_namedEntities.insert ( "varpropto", "221D" );
  m_namedEntities.insert ( "varr", "2195" );
  m_namedEntities.insert ( "varrho", "03F1" );
  m_namedEntities.insert ( "varsigma", "03C2" );
  m_namedEntities.insert ( "varsubsetneq", "FE00" );
  m_namedEntities.insert ( "varsubsetneqq", "FE00" );
  m_namedEntities.insert ( "varsupsetneq", "FE00" );
  m_namedEntities.insert ( "varsupsetneqq", "FE00" );
  m_namedEntities.insert ( "vartheta", "03D1" );
  m_namedEntities.insert ( "vartriangleleft", "22B2" );
  m_namedEntities.insert ( "vartriangleright", "22B3" );
  m_namedEntities.insert ( "vcy", "0432" );
  m_namedEntities.insert ( "vdash", "22A2" );
  m_namedEntities.insert ( "vee", "2228" );
  m_namedEntities.insert ( "veebar", "22BB" );
  m_namedEntities.insert ( "veeeq", "225A" );
  m_namedEntities.insert ( "vellip", "22EE" );
  m_namedEntities.insert ( "verbar", "007C" );
  m_namedEntities.insert ( "vert", "007C" );
  m_namedEntities.insert ( "vfr", "DD33" );
  m_namedEntities.insert ( "vltri", "22B2" );
  m_namedEntities.insert ( "vnsub", "20D2" );
  m_namedEntities.insert ( "vnsup", "20D2" );
  m_namedEntities.insert ( "vopf", "DD67" );
  m_namedEntities.insert ( "vprop", "221D" );
  m_namedEntities.insert ( "vrtri", "22B3" );
  m_namedEntities.insert ( "vscr", "DCCB" );
  m_namedEntities.insert ( "vsubnE", "FE00" );
  m_namedEntities.insert ( "vsubne", "FE00" );
  m_namedEntities.insert ( "vsupnE", "FE00" );
  m_namedEntities.insert ( "vsupne", "FE00" );
  m_namedEntities.insert ( "vzigzag", "299A" );
  m_namedEntities.insert ( "wcirc", "0175" );
  m_namedEntities.insert ( "wedbar", "2A5F" );
  m_namedEntities.insert ( "wedge", "2227" );
  m_namedEntities.insert ( "wedgeq", "2259" );
  m_namedEntities.insert ( "weierp", "2118" );
  m_namedEntities.insert ( "wfr", "DD34" );
  m_namedEntities.insert ( "wopf", "DD68" );
  m_namedEntities.insert ( "wp", "2118" );
  m_namedEntities.insert ( "wr", "2240" );
  m_namedEntities.insert ( "wreath", "2240" );
  m_namedEntities.insert ( "wscr", "DCCC" );
  m_namedEntities.insert ( "xcap", "22C2" );
  m_namedEntities.insert ( "xcirc", "25EF" );
  m_namedEntities.insert ( "xcup", "22C3" );
  m_namedEntities.insert ( "xdtri", "25BD" );
  m_namedEntities.insert ( "xfr", "DD35" );
  m_namedEntities.insert ( "xhArr", "27FA" );
  m_namedEntities.insert ( "xharr", "27F7" );
  m_namedEntities.insert ( "xi", "03BE" );
  m_namedEntities.insert ( "xlArr", "27F8" );
  m_namedEntities.insert ( "xlarr", "27F5" );
  m_namedEntities.insert ( "xmap", "27FC" );
  m_namedEntities.insert ( "xnis", "22FB" );
  m_namedEntities.insert ( "xodot", "2A00" );
  m_namedEntities.insert ( "xopf", "DD69" );
  m_namedEntities.insert ( "xoplus", "2A01" );
  m_namedEntities.insert ( "xotime", "2A02" );
  m_namedEntities.insert ( "xrArr", "27F9" );
  m_namedEntities.insert ( "xrarr", "27F6" );
  m_namedEntities.insert ( "xscr", "DCCD" );
  m_namedEntities.insert ( "xsqcup", "2A06" );
  m_namedEntities.insert ( "xuplus", "2A04" );
  m_namedEntities.insert ( "xutri", "25B3" );
  m_namedEntities.insert ( "xvee", "22C1" );
  m_namedEntities.insert ( "xwedge", "22C0" );
  m_namedEntities.insert ( "yacute", "00FD" );
  m_namedEntities.insert ( "yacy", "044F" );
  m_namedEntities.insert ( "ycirc", "0177" );
  m_namedEntities.insert ( "ycy", "044B" );
  m_namedEntities.insert ( "yen", "00A5" );
  m_namedEntities.insert ( "yfr", "DD36" );
  m_namedEntities.insert ( "yicy", "0457" );
  m_namedEntities.insert ( "yopf", "DD6A" );
  m_namedEntities.insert ( "yscr", "DCCE" );
  m_namedEntities.insert ( "yucy", "044E" );
  m_namedEntities.insert ( "yuml", "00FF" );
  m_namedEntities.insert ( "zacute", "017A" );
  m_namedEntities.insert ( "zcaron", "017E" );
  m_namedEntities.insert ( "zcy", "0437" );
  m_namedEntities.insert ( "zdot", "017C" );
  m_namedEntities.insert ( "zeetrf", "2128" );
  m_namedEntities.insert ( "zeta", "03B6" );
  m_namedEntities.insert ( "zfr", "DD37" );
  m_namedEntities.insert ( "zhcy", "0436" );
  m_namedEntities.insert ( "zigrarr", "21DD" );
  m_namedEntities.insert ( "zopf", "DD6B" );
  m_namedEntities.insert ( "zscr", "DCCF" );
  m_namedEntities.insert ( "zwj", "200D" );
  m_namedEntities.insert ( "zwnj", "200C" );

}
